import unittest
from unittest.mock import patch

import math

from drone_base.config.drone import GimbalType
from drone_base.control.operations import TiltCommand, MovementByCommand, PilotingCommand, CommandSequence, Movement, \
    MovementCommandFactory, PilotingCommandFactory


class TestTiltCommand(unittest.TestCase):
    def test_init_default_values(self):
        """Test TiltCommand initialization with default values."""
        cmd = TiltCommand(pitch_deg=45.0)
        self.assertEqual(cmd.pitch_deg, 45.0)
        self.assertEqual(cmd.control_mode, GimbalType.MODE_POSITION)
        self.assertEqual(cmd.reference_type, GimbalType.REF_ABSOLUTE)

    def test_init_custom_values(self):
        """Test TiltCommand initialization with custom values."""
        cmd = TiltCommand(
            pitch_deg=-30.0,
            control_mode=GimbalType.MODE_VELOCITY,
            reference_type=GimbalType.REF_RELATIVE
        )
        self.assertEqual(cmd.pitch_deg, -30.0)
        self.assertEqual(cmd.control_mode, GimbalType.MODE_VELOCITY)
        self.assertEqual(cmd.reference_type, GimbalType.REF_RELATIVE)


class TestMovementByCommand(unittest.TestCase):
    def test_init_default_values(self):
        """Test MovementByCommand initialization with default values."""
        cmd = MovementByCommand()
        self.assertEqual(cmd.forward, 0)
        self.assertEqual(cmd.right, 0)
        self.assertEqual(cmd.down, 0)
        self.assertEqual(cmd.rotation, 0)
        self.assertFalse(cmd.is_random_movement)
        self.assertFalse(cmd.will_also_land_command)

    def test_init_custom_values(self):
        """Test MovementByCommand initialization with custom values."""
        cmd = MovementByCommand(
            forward=1.5,
            right=-0.5,
            down=0.2,
            rotation=math.pi / 2,
            is_random_movement=True,
            will_also_land_command=True
        )
        self.assertEqual(cmd.forward, 1.5)
        self.assertEqual(cmd.right, -0.5)
        self.assertEqual(cmd.down, 0.2)
        self.assertEqual(cmd.rotation, math.pi / 2)
        self.assertTrue(cmd.is_random_movement)
        self.assertTrue(cmd.will_also_land_command)


class TestPilotingCommand(unittest.TestCase):
    def test_init_default_values(self):
        """Test PilotingCommand initialization with default values."""
        cmd = PilotingCommand()
        self.assertEqual(cmd.x, 0)
        self.assertEqual(cmd.y, 0)
        self.assertEqual(cmd.z, 0)
        self.assertEqual(cmd.rotation, 0)
        self.assertEqual(cmd.dt, 0.15)
        self.assertFalse(cmd.is_random_movement)
        self.assertFalse(cmd.will_also_land_command)

    def test_init_custom_values(self):
        """Test PilotingCommand initialization with custom values."""
        cmd = PilotingCommand(
            x=10,
            y=20,
            z=-5,
            rotation=90,
            dt=0.25,
            is_random_movement=True,
            will_also_land_command=True
        )
        self.assertEqual(cmd.x, 10)
        self.assertEqual(cmd.y, 20)
        self.assertEqual(cmd.z, -5)
        self.assertEqual(cmd.rotation, 90)
        self.assertEqual(cmd.dt, 0.25)
        self.assertTrue(cmd.is_random_movement)
        self.assertTrue(cmd.will_also_land_command)


class TestCommandSequence(unittest.TestCase):
    def test_init(self):
        """Test CommandSequence initialization."""
        commands = [
            MovementByCommand(forward=0.5),
            TiltCommand(pitch_deg=30.0)
        ]
        sequence = CommandSequence(commands=commands)
        self.assertEqual(sequence.commands, commands)


class TestMovementCommandFactory(unittest.TestCase):
    def setUp(self):
        """Set up the factory before each test."""
        self.factory = MovementCommandFactory()

    def test_init_default_values(self):
        """Test MovementCommandFactory initialization with default values."""
        self.assertEqual(self.factory.forward_speed, 0.3)
        self.assertEqual(self.factory.rotation_speed, math.pi / 3)
        self.assertEqual(self.factory.pi_over_two, math.pi / 2)
        self.assertEqual(self.factory._possible_turns, [math.pi / 3, -math.pi / 3])

    def test_init_custom_values(self):
        """Test MovementCommandFactory initialization with custom values."""
        factory = MovementCommandFactory(forward_speed=0.5)
        self.assertEqual(factory.forward_speed, 0.5)

    def test_get_command_forward(self):
        """Test get_command for FORWARD movement."""
        cmd = self.factory.get_command(Movement.FORWARD)
        self.assertEqual(cmd.forward, 0.3)
        self.assertEqual(cmd.right, 0)
        self.assertEqual(cmd.down, 0)
        self.assertEqual(cmd.rotation, 0)
        self.assertFalse(cmd.is_random_movement)

    def test_get_command_turn_left(self):
        """Test get_command for TURN_LEFT movement."""
        cmd = self.factory.get_command(Movement.TURN_LEFT)
        self.assertEqual(cmd.forward, 0)
        self.assertEqual(cmd.right, 0)
        self.assertEqual(cmd.down, 0)
        self.assertEqual(cmd.rotation, -math.pi / 3)
        self.assertFalse(cmd.is_random_movement)

    def test_get_command_turn_right(self):
        """Test get_command for TURN_RIGHT movement."""
        cmd = self.factory.get_command(Movement.TURN_RIGHT)
        self.assertEqual(cmd.forward, 0)
        self.assertEqual(cmd.right, 0)
        self.assertEqual(cmd.down, 0)
        self.assertEqual(cmd.rotation, math.pi / 3)
        self.assertFalse(cmd.is_random_movement)

    def test_get_command_u_turn(self):
        """Test get_command for U_TURN movement."""
        cmd = self.factory.get_command(Movement.U_TURN)
        self.assertEqual(cmd.forward, 0)
        self.assertEqual(cmd.right, 0)
        self.assertEqual(cmd.down, 0)
        self.assertEqual(cmd.rotation, math.pi / 2)
        self.assertFalse(cmd.is_random_movement)

    @patch('random.choice')
    def test_get_command_random(self, mock_choice):
        """Test get_command for RANDOM movement."""
        mock_choice.return_value = math.pi / 3

        cmd = self.factory.get_command(Movement.RANDOM)
        self.assertEqual(cmd.forward, 0)
        self.assertEqual(cmd.right, 0)
        self.assertEqual(cmd.down, 0)
        self.assertEqual(cmd.rotation, math.pi / 3)
        self.assertTrue(cmd.is_random_movement)

        mock_choice.assert_called_once_with([math.pi / 3, -math.pi / 3])


class TestPilotingCommandFactory(unittest.TestCase):
    def setUp(self):
        """Set up the factory before each test."""
        self.factory = PilotingCommandFactory()

    def test_init_default_values(self):
        """Test PilotingCommandFactory initialization with default values."""
        self.assertEqual(self.factory.forward_speed, 35)
        self.assertEqual(self.factory.rotation_speed, 35)
        self.assertEqual(self.factory.rotation_dt, 0.5)
        self.assertEqual(self.factory._possible_turns, [35, -35])

    def test_init_custom_values(self):
        """Test PilotingCommandFactory initialization with custom values."""
        factory = PilotingCommandFactory(forward_speed=50, rotation_speed=40)
        self.assertEqual(factory.forward_speed, 50)
        self.assertEqual(factory.rotation_speed, 40)

    def test_get_piloting_command_forward(self):
        """Test get_piloting_command for FORWARD movement."""
        cmd = self.factory.get_piloting_command(Movement.FORWARD)
        self.assertEqual(cmd.x, 0)
        self.assertEqual(cmd.y, 35)
        self.assertEqual(cmd.z, 0)
        self.assertEqual(cmd.rotation, 0)
        self.assertEqual(cmd.dt, 0.15)
        self.assertFalse(cmd.is_random_movement)

    def test_get_piloting_command_turn_left(self):
        """Test get_piloting_command for TURN_LEFT movement."""
        cmd = self.factory.get_piloting_command(Movement.TURN_LEFT)
        self.assertEqual(cmd.x, 0)
        self.assertEqual(cmd.y, 0)
        self.assertEqual(cmd.z, 0)
        self.assertEqual(cmd.rotation, -35)
        self.assertEqual(cmd.dt, 0.5)
        self.assertFalse(cmd.is_random_movement)

    @patch('random.choice')
    def test_get_piloting_command_random(self, mock_choice):
        """Test get_piloting_command for RANDOM movement."""
        mock_choice.return_value = 35

        cmd = self.factory.get_piloting_command(Movement.RANDOM)
        self.assertEqual(cmd.x, 0)
        self.assertEqual(cmd.y, 0)
        self.assertEqual(cmd.z, 0)
        self.assertEqual(cmd.rotation, 35)
        self.assertEqual(cmd.dt, 0.5)
        self.assertTrue(cmd.is_random_movement)

        mock_choice.assert_called_once_with([35, -35])


if __name__ == '__main__':
    unittest.main()
