import logging
import sys
from logging.handlers import RotatingFileHandler
from pathlib import Path
from typing import Final


class LoggerSetup:
    LEVEL_WIDTH: Final[int] = 8

    @staticmethod
    def setup_logger(
            logger_name: str,
            log_file: str | Path | None = None,
            level: int = logging.DEBUG,
            console_level: int = logging.INFO,
            file_level: int = logging.DEBUG
    ) -> logging.Logger:
        """
        Set up a logger with both console and file handlers.

        :param logger_name: Name of the logger.
        :param log_file: Optional path to log file. If None, only console logging is set.
        :param level: Overall logging level.
        :param console_level: Logging level for console output.
        :param file_level: Logging level for file output.
        :return: Configured logger instance.
        """
        logger = logging.getLogger(logger_name)
        logger.setLevel(level)
        logger.handlers.clear()

        console_formatter = logging.Formatter(
            f'%(asctime)s - [%(levelname)-{LoggerSetup.LEVEL_WIDTH:d}s] - %(name)s - '
            f'[%(filename)s:%(lineno)d] - %(message)s'
        )
        file_formatter = logging.Formatter(
            f'%(asctime)s - [%(levelname)-{LoggerSetup.LEVEL_WIDTH:d}s] - %(name)s - [%(filename)s:%(lineno)d] - '
            f'[Thread: %(threadName)s | PID: %(process)d] - %(message)s'
        )

        console_handler = logging.StreamHandler(sys.stdout)
        console_handler.setLevel(console_level)
        console_handler.setFormatter(console_formatter)
        logger.addHandler(console_handler)

        if log_file:
            log_path = Path(log_file)
            log_path.parent.mkdir(parents=True, exist_ok=True)

            file_handler = RotatingFileHandler(log_file, maxBytes=5 * 1024 * 1024, backupCount=5)
            file_handler.setLevel(file_level)
            file_handler.setFormatter(file_formatter)
            logger.addHandler(file_handler)

        return logger
