from pathlib import Path
from typing import Final

from pynput.keyboard import KeyCode, Listener

from drone_base.config.drone import DroneSpeed, GimbalType
from drone_base.config.logger import LoggerSetup
from drone_base.control.drone_commander import DroneCommander
from drone_base.control.manual.operations import DroneCommand, MovementVector


class KeyboardController:
    """
    Handles keyboard input for drone control with support for multiple simultaneous inputs.

    The Movement keys are:
        - i - Move forward.
        - k - Move backward.
        - j - Move right.
        - l - Move left.
        - o - Rotate right.
        - u - Rotate left.
        - n - Increase height.
        - m - Decrease height.

    The Command keys are:
        - q - Stop the drone.
        - t - Start Take off.
        - y - Land the drone.
        - p - Takes a photo.
        - z - Resets the gimbal camera position.
        - x - Tilts the gimbal camera position 15 degrees.
        - c - Tilts the gimbal camera position 30 degrees.
        - v - Tilts the gimbal camera position 45 degrees.
        - s - Tilts the gimbal camera position 60 degrees.
        - d - Tilts the gimbal camera position 75 degrees.
        - f - Tilts the gimbal camera position 90 degrees.
    """

    MOVEMENT_KEYS: Final[dict[str, tuple[str, int]]] = {
        "i": ("y", 1),
        "k": ("y", -1),
        "j": ("x", 1),
        "l": ("x", -1),
        "o": ("z_rot", 1),
        "u": ("z_rot", -1),
        "n": ("z", 1),
        "m": ("z", -1),
    }

    COMMAND_KEYS: Final[dict[str, DroneCommand]] = {
        "t": DroneCommand.TAKEOFF,
        "y": DroneCommand.LAND,
        "p": DroneCommand.PHOTO,
        "z": DroneCommand.TILT_RESET,
        "x": DroneCommand.TILT_15,
        "c": DroneCommand.TILT_30,
        "v": DroneCommand.TILT_45,
        "s": DroneCommand.TILT_60,
        "d": DroneCommand.TILT_75,
        "f": DroneCommand.TILT_90,
        "q": DroneCommand.EMERGENCY_STOP,
    }

    def __init__(
            self,
            commander: DroneCommander,
            speed_config: DroneSpeed = DroneSpeed(),
            logger_dir: str | Path | None = None,
    ):
        """
        :param commander: The DroneCommander instance that will handle all drone operations.
        :param speed_config: The speed configuration class of the drone. Contains speed details,
                             if not given it will use default speeds.
        """
        # TODO: find a way to work around drone commander. ISSUE: Using ANAFI-AI (works on ANAFI4K)
        #  drone it doesn't override the autonomous control command with the manual command
        self.commander = commander
        self.speed_config = speed_config

        self.movement_vector = MovementVector()
        self.listener: Listener | None = None
        self.is_running = False
        if logger_dir is not None:
            logger_dir = Path(logger_dir) / f"{self.__class__.__name__}.log"
        self.logger = LoggerSetup.setup_logger(logger_name=self.__class__.__name__, log_file=logger_dir)

    def start(self):
        """Start listening for keyboard input"""
        self.is_running = True
        if self.listener is None:
            self.listener = Listener(
                on_press=self._on_press_button,
                on_release=self._on_release_button
            )
            self.listener.start()
        self.logger.info("Started keyboard control thread")

    def stop(self):
        """Stop listening for keyboard input"""
        self.is_running = False
        if self.listener is not None:
            self.listener.stop()
            self.listener = None
            self.__handle_emergency_stop()
            self.logger.info("Ended keyboard control thread")

    def _on_press_button(self, key: KeyCode):
        """Handle key press events."""
        if not hasattr(key, "char"):
            return
        char = key.char

        if char in self.COMMAND_KEYS:
            self._execute_command(self.COMMAND_KEYS[char])
            return

        if char in self.MOVEMENT_KEYS:
            axis, direction = self.MOVEMENT_KEYS[char]
            setattr(self.movement_vector, axis, direction * self.speed_config.__getattribute__(axis))
            self._update_movement()

    def _on_release_button(self, key: KeyCode):
        """Handles key release"""
        if not hasattr(key, "char"):
            return
        char = key.char
        if char in self.MOVEMENT_KEYS:
            axis, _ = self.MOVEMENT_KEYS[char]
            setattr(self.movement_vector, axis, 0)
            self._update_movement()

    def _update_movement(self):
        if self.movement_vector.is_moving():
            self.commander.piloting(
                x=self.movement_vector.x,
                y=self.movement_vector.y,
                z_rot=self.movement_vector.z_rot,
                z=self.movement_vector.z,
                dt=self.speed_config.dt
            )

    def _execute_command(self, command: DroneCommand):
        """Handles given command."""
        try:
            if command == DroneCommand.TAKEOFF:
                self.commander.take_off()
            elif command == DroneCommand.LAND:
                self.commander.land()
            elif command == DroneCommand.PHOTO:
                self.logger.info("Photo taken.")
            elif command == DroneCommand.TILT_RESET:
                self.commander.tilt_camera(pitch_deg=0, reference_type=GimbalType.REF_ABSOLUTE)
            elif command == DroneCommand.TILT_15:
                self.commander.tilt_camera(pitch_deg=-15, reference_type=GimbalType.REF_ABSOLUTE)
            elif command == DroneCommand.TILT_30:
                self.commander.tilt_camera(pitch_deg=-30, reference_type=GimbalType.REF_ABSOLUTE)
            elif command == DroneCommand.TILT_45:
                self.commander.tilt_camera(pitch_deg=-45, reference_type=GimbalType.REF_ABSOLUTE)
            elif command == DroneCommand.TILT_60:
                self.commander.tilt_camera(pitch_deg=-60, reference_type=GimbalType.REF_ABSOLUTE)
            elif command == DroneCommand.TILT_75:
                self.commander.tilt_camera(pitch_deg=-75, reference_type=GimbalType.REF_ABSOLUTE)
            elif command == DroneCommand.TILT_90:
                self.commander.tilt_camera(pitch_deg=-90, reference_type=GimbalType.REF_ABSOLUTE)
            elif command == DroneCommand.EMERGENCY_STOP:
                self.stop()
        except Exception as e:
            self.logger.error("Encountered exception for command: %s", command, exc_info=e)

    def __handle_emergency_stop(self):
        self.movement_vector.reset()
        self._update_movement()
        self.commander.land()
