import queue
import unittest
from unittest.mock import Mock, patch

import numpy as np
import olympe

from drone_base.config.video import VideoConfig
from drone_base.stream.stream_handler import StreamHandler


class TestStreamHandler(unittest.TestCase):
    def setUp(self):
        """Set up test fixtures before each test method."""
        self.mock_drone = Mock(spec=olympe.Drone)
        self.mock_drone.streaming = Mock()

        self.video_config = VideoConfig(width=1280, height=720)
        self.video_config.max_queue_size = 10

        self.stream_handler = StreamHandler(
            drone=self.mock_drone,
            video_config=self.video_config,
            is_renderer=False
        )

    def test_init(self):
        """Test the initialization of StreamHandler."""
        self.assertEqual(self.stream_handler.config.max_queue_size, 10)
        self.assertEqual(self.stream_handler.target_size, (1280, 720))
        self.assertIsInstance(self.stream_handler.frame_queue, queue.Queue)
        self.assertEqual(self.stream_handler.frame_queue.maxsize, 10)
        self.assertFalse(self.stream_handler.is_renderer_started)

    def test_start_streaming(self):
        """Test the start_streaming method."""
        self.stream_handler.start_streaming()
        self.mock_drone.streaming.set_callbacks.assert_called_once()
        self.mock_drone.streaming.start.assert_called_once()
        self.assertIsNone(self.stream_handler.renderer)

    @patch("cv2.destroyAllWindows")
    def test_stop_streaming(self, mock_destroy_windows):
        """Test the stop_streaming method."""
        self.stream_handler.stop_streaming()

        self.mock_drone.streaming.stop.assert_called_once()
        mock_destroy_windows.assert_called_once()

    @patch("cv2.resize")
    @patch("cv2.cvtColor")
    def test_yuv_to_cv2_frame(self, mock_cvt_color, mock_resize):
        """Test YUV frame conversion to CV2 frame."""
        mock_yuv_frame = Mock(spec=olympe.VideoFrame)
        mock_yuv_frame.format.return_value = olympe.VDEF_I420
        mock_yuv_frame.as_ndarray.return_value = np.zeros((720, 1280, 3))

        mock_cvt_color.return_value = np.zeros((720, 1280, 3))
        mock_resize.return_value = np.zeros((720, 1280, 3))

        result = self.stream_handler.yuv_to_cv2_frame(mock_yuv_frame)

        mock_yuv_frame.as_ndarray.assert_called_once()
        mock_cvt_color.assert_called_once()
        mock_resize.assert_called_once()
        self.assertIsInstance(result, np.ndarray)

    def test_yuv_frame_cb(self):
        """Test YUV frame callback."""
        mock_yuv_frame = Mock(spec=olympe.VideoFrame)
        mock_yuv_frame.format.return_value = olympe.VDEF_I420

        with patch.object(self.stream_handler, "yuv_to_cv2_frame") as mock_convert:
            mock_convert.return_value = np.zeros((720, 1280, 3))
            self.stream_handler.yuv_frame_cb(mock_yuv_frame)

            mock_yuv_frame.ref.assert_called_once()
            mock_yuv_frame.unref.assert_called_once()
            self.assertEqual(self.stream_handler.frame_queue.qsize(), 1)

    def test_flush_cb(self):
        """Test flush callback."""
        for _ in range(3):
            self.stream_handler.frame_queue.put(np.zeros((720, 1280, 3)))

        mock_stream = {"vdef_format": olympe.VDEF_I420}
        result = self.stream_handler.flush_cb(mock_stream)

        self.assertTrue(result)
        self.assertTrue(self.stream_handler.frame_queue.empty())

    def test_stop_streaming_with_renderer(self):
        """Test stop_streaming with renderer enabled."""
        mock_renderer = Mock()
        self.stream_handler.renderer = mock_renderer
        self.stream_handler.stop_streaming()

        mock_renderer.stop.assert_called_once()
        self.mock_drone.streaming.stop.assert_called_once()

    def test_stop_streaming_with_exception(self):
        """Test stop_streaming when an exception occurs."""
        mock_renderer = Mock()
        mock_renderer.stop.side_effect = Exception("Test exception")
        self.stream_handler.renderer = mock_renderer

        self.stream_handler.stop_streaming()
        mock_renderer.stop.assert_called_once()


if __name__ == "__main__":
    unittest.main()
