import queue
import threading
import time
from abc import ABC, abstractmethod
from contextlib import contextmanager
from pathlib import Path

import cv2
import numpy as np

from drone_base.config.logger import LoggerSetup
from drone_base.config.video import VideoConfig
from drone_base.control.drone_commander import DroneCommander
from drone_base.stream.saving.frame_saver import BufferedFrameSaver


class BaseVideoProcessor(ABC):
    def __init__(
            self,
            video_config: VideoConfig,
            frame_queue: queue.Queue,
            drone_commander: DroneCommander,
            logger_dir: str | Path | None = None,
            frame_save_dir: str | Path | None = None,
    ):
        self.config = video_config
        self.frame_queue = frame_queue
        self.drone_commander = drone_commander

        self._running = threading.Event()
        self._lock = threading.RLock()
        self._frame_count = 0

        self.is_frame_saved = frame_save_dir is not None
        if self.is_frame_saved:
            save_path = Path(frame_save_dir) / "frames"
            self.frame_saver = BufferedFrameSaver(
                output_dir=save_path, logger_dir=logger_dir, save_extension=video_config.save_extension
            )

        if logger_dir is not None:
            logger_dir = Path(logger_dir) / f"{self.__class__.__name__}.log"
        self.logger = LoggerSetup.setup_logger(logger_name=self.__class__.__name__, log_file=logger_dir)

    def start(self):
        """Start the frame processing loop in a daemon thread."""
        self._running.set()
        thread = threading.Thread(
            target=self._run_processing_loop, daemon=True, name=f"{self.__class__.__name__}Thread"
        )
        thread.start()
        self.logger.info("%s Frame processing thread started.", self.__class__.__name__)

    def stop(self):
        """Stop the frame processing loop."""
        self._running.clear()
        self.logger.info("%s Frame processing stop signal sent.", self.__class__.__name__)
        if self.is_frame_saved:
            self.frame_saver.save_all()

    @contextmanager
    def _frame_display_context(self):
        """Context manager for safely handling frame display resources."""
        try:
            yield
        finally:
            cv2.destroyAllWindows()

    @abstractmethod
    def _process_frame(self, frame: np.ndarray) -> np.ndarray:
        """Process a single frame with all required modifications."""
        raise NotImplementedError("This method must be implemented.")

    @staticmethod
    def _display_frame(cv2_frame: np.ndarray) -> None:
        """This function will be called if we have frames in self.frame_queue."""
        cv2.imshow("Parrot Olympe Video Stream", cv2_frame)
        cv2.waitKey(1)

    def _run_processing_loop(self):
        self.logger.info("Starting frame processing loop.")

        with self._frame_display_context():
            while self._running.is_set() and threading.main_thread().is_alive():
                try:
                    frame = self.frame_queue.get(timeout=0.1)
                    with self._lock:
                        self._frame_count += 1
                        if self.is_frame_saved:
                            self.frame_saver.add_frame(frame=np.array(frame), timestamp=time.perf_counter())
                        self._display_frame(self._process_frame(frame))
                except queue.Empty:
                    continue
                except Exception as e:
                    self.logger.error("Unable to process frame...")
                    self.logger.critical(e, exc_info=True)
                    if not self._running.is_set():
                        break

        self.logger.info("Frame processing loop terminated.")
