import queue
import threading
import time
from pathlib import Path

import cv2
import numpy as np
import olympe
from olympe import PdrawRenderer
from olympe.messages import camera

from drone_base.config.logger import LoggerSetup
from drone_base.config.video import VideoConfig
from drone_base.stream.processing.streaming_metadata import save_data


# TODO: maybe context manager?
class StreamHandler:
    """
    Handler for drone video streams that processes frames and manages metadata.

    This class handles the streaming of video from a drone, converting frames to OpenCV
    format, and optionally saving metadata associated with the stream.
    """

    def __init__(self, drone: olympe.Drone, video_config: VideoConfig, is_renderer: bool = False,
                 logger_dir: str | Path | None = None, metadata_dir: str | Path | None = None):
        self.drone = drone
        self.config = video_config
        self.is_renderer_started = is_renderer
        self.renderer = None

        if logger_dir is not None:
            logger_dir = Path(logger_dir) / f"{self.__class__.__name__}.log"
        self.logger = LoggerSetup.setup_logger(logger_name=self.__class__.__name__, log_file=logger_dir)

        self.frame_queue = queue.Queue(maxsize=self.config.max_queue_size)
        self.flush_queue_lock = threading.RLock()

        self.target_size = (self.config.width, self.config.height)
        self.cv2_cvt_colors = {
            olympe.VDEF_I420: cv2.COLOR_YUV2BGR_I420,
            olympe.VDEF_NV12: cv2.COLOR_YUV2BGR_NV12,
        }

        self.is_metadata_saved: bool = metadata_dir is not None
        self.metadata = []
        if metadata_dir is not None:
            metadata_dir = Path(metadata_dir) / "metadata.json"
        self.metadata_save_path = metadata_dir

        self.is_streaming = False

    def start_streaming(self):
        """Setup callback functions for live config processing and starts the config streaming."""
        self.drone.streaming.set_callbacks(
            raw_cb=self.yuv_frame_cb,
            h264_cb=self.h264_frame_cb,
            start_cb=self.start_cb,
            end_cb=self.end_cb,
            flush_raw_cb=self.flush_cb,
        )
        self.logger.info("Starting streaming...")
        self.drone.streaming.start()

        if self.config.cam_mode in ("photo", "recording"):
            self.drone(camera.set_camera_mode(cam_id=0, value=self.config.cam_mode))

        self.is_streaming = True
        if self.is_renderer_started:
            self.renderer = PdrawRenderer(pdraw=self.drone.streaming)

    def stop_streaming(self):
        """Properly stop the config stream and disconnect."""
        self.logger.info("Stopping streaming...")
        try:
            if self.renderer is not None:
                self.renderer.stop()
                self.renderer = None
            if self.drone is not None:
                self.drone.streaming.stop()
            if self.is_metadata_saved:
                self._save_metadata()
        except Exception as e:
            self.logger.error("Unable to properly stop the streaming...")
            self.logger.critical(e, exc_info=True)
        finally:
            self.is_streaming = False
            cv2.destroyAllWindows()

    def yuv_to_cv2_frame(self, yuv_frame: olympe.VideoFrame) -> cv2.Mat | np.ndarray:
        """
        This function will process the YUV frame and convert it into OpenCV format then resizing it to the values given
        in the constructor.
        """
        return cv2.resize(
            cv2.cvtColor(yuv_frame.as_ndarray(), self.cv2_cvt_colors[yuv_frame.format()]),
            self.target_size, interpolation=cv2.INTER_LINEAR
        )

    def yuv_frame_cb(self, yuv_frame: olympe.VideoFrame):
        """
        This function will be called by Olympe for each decoded YUV frame. It transforms the YUV frame into an OpenCV
        frame, and unrefs the frame.
        """
        if not yuv_frame:
            self.logger.warning("Received empty frame")
            return
        try:
            yuv_frame.ref()
            self.frame_queue.put_nowait(self.yuv_to_cv2_frame(yuv_frame))
            if self.is_metadata_saved:
                self._prepare_metadata(vmeta_data=yuv_frame.vmeta()[1])
        except queue.Full:
            self.logger.warning("Queue is full")
        finally:
            yuv_frame.unref()

    def flush_cb(self, stream):
        """Handle stream flush events."""
        self.logger.warning("Flush requested for stream. Resetting queue.")
        if stream["vdef_format"] != olympe.VDEF_I420:
            return True
        with self.flush_queue_lock:
            while not self.frame_queue.empty():
                self.frame_queue.get_nowait()
        return True

    def start_cb(self):
        self.logger.info("Video stream started.")

    def end_cb(self):
        self.logger.info("Video stream ended.")

    def h264_frame_cb(self, h264_frame):
        pass

    def _prepare_metadata(self, vmeta_data: dict) -> None:
        if len(vmeta_data) == 0:
            return
        metadata_ = {
            "time": time.time(),
            "drone": vmeta_data["drone"],
            "camera": vmeta_data["camera"],
        }
        self.metadata.append(metadata_)

    def _save_metadata(self) -> None:
        """Save the current metadata to disk and clear the in-memory list."""
        if self.metadata_save_path is None or len(self.metadata) == 0:
            self.logger.warning("No metadata to save or path non existent.")
            return

        try:
            self.logger.info("Saving %s metadata entries...", len(self.metadata))
            save_data(file_path=self.metadata_save_path, data=self.metadata)
            self.metadata.clear()
        except Exception as e:
            self.logger.error("Unable to save metadata entries...")
            self.logger.critical(e, exc_info=True)
