import unittest

from drone_base.config.drone import DroneSpeed, GimbalType, DroneIp


class TestDroneSpeed(unittest.TestCase):
    def test_default_initialization(self):
        """Test that DroneSpeed initializes with default values correctly"""
        drone_speed = DroneSpeed()
        expected_defaults = {
            "x": 20,
            "y": 20,
            "z": 20,
            "z_rot": 20,
            "dt": 0.1
        }

        for attr, expected_value in expected_defaults.items():
            with self.subTest(attr=attr):
                self.assertEqual(getattr(drone_speed, attr), expected_value)

    def test_valid_custom_values(self):
        """Test initialization with valid custom values"""
        custom_values = {
            "x": 50,
            "y": 75,
            "z": 25,
            "z_rot": 100,
            "dt": 0.5
        }

        drone_speed = DroneSpeed(**custom_values)

        for attr, expected_value in custom_values.items():
            with self.subTest(attr=attr):
                self.assertEqual(getattr(drone_speed, attr), expected_value)

    def test_invalid_range_values(self):
        """Test that values outside the valid range raise ValueError with correct message"""
        range_attrs = ["x", "y", "z", "z_rot"]
        test_cases = []

        for attr in range_attrs:
            test_cases.extend([
                (attr, -1, "must be between [0, 100], got \"-1\""),
                (attr, 101, "must be between [0, 100], got \"101\"")
            ])

        for attr_name, invalid_value, expected_msg in test_cases:
            with self.subTest(attr=attr_name, value=invalid_value):
                kwargs = {attr_name: invalid_value}
                with self.assertRaises(ValueError) as context:
                    DroneSpeed(**kwargs)
                self.assertIn(expected_msg, str(context.exception))

    def test_invalid_dt(self):
        """Test that negative dt raises ValueError"""
        with self.assertRaises(ValueError) as context:
            DroneSpeed(dt=-0.1)
        self.assertIn("Time step must be positive, got \"-0.1\"", str(context.exception))

    def test_boundary_values(self):
        """Test that boundary values are accepted"""
        boundary_cases = [
            # (test_name, kwargs)
            ("lower_bounds", {"x": 0, "y": 0, "z": 0, "z_rot": 0, "dt": 0}),
            ("upper_bounds", {"x": 100, "y": 100, "z": 100, "z_rot": 100, "dt": 100})
        ]

        for test_name, kwargs in boundary_cases:
            with self.subTest(case=test_name):
                drone_speed = DroneSpeed(**kwargs)
                for attr, value in kwargs.items():
                    self.assertEqual(getattr(drone_speed, attr), value)

    def test_partial_initialization(self):
        """Test that partial initialization works with defaults"""
        partial_values = {"x": 75, "z_rot": 90}
        expected_values = {
            "x": 75,
            "y": 20,  # default
            "z": 20,  # default
            "z_rot": 90,
            "dt": 0.1  # default
        }

        drone_speed = DroneSpeed(**partial_values)

        for attr, expected_value in expected_values.items():
            with self.subTest(attr=attr):
                self.assertEqual(getattr(drone_speed, attr), expected_value)


class TestGimbalType(unittest.TestCase):
    def test_enum_values(self):
        self.assertEqual(GimbalType.REF_ABSOLUTE.value, "absolute")
        self.assertEqual(GimbalType.REF_RELATIVE.value, "relative")
        self.assertEqual(GimbalType.MODE_POSITION.value, "position")
        self.assertEqual(GimbalType.MODE_VELOCITY.value, "velocity")

    def test_str_method(self):
        self.assertEqual(str(GimbalType.REF_ABSOLUTE), "absolute")
        self.assertEqual(str(GimbalType.REF_RELATIVE), "relative")
        self.assertEqual(str(GimbalType.MODE_POSITION), "position")
        self.assertEqual(str(GimbalType.MODE_VELOCITY), "velocity")

    def test_enum_members(self):
        expected_values = {"absolute", "relative", "position", "velocity"}
        enum_values = {member.value for member in GimbalType}
        self.assertEqual(enum_values, expected_values)


class TestDroneIp(unittest.TestCase):
    def test_enum_values(self):
        self.assertEqual(DroneIp.CABLE.value, "192.168.53.1")
        self.assertEqual(DroneIp.WIRELESS.value, "192.168.42.1")
        self.assertEqual(DroneIp.SIMULATED.value, "10.202.0.1")

    def test_str_method(self):
        self.assertEqual(str(DroneIp.CABLE), "192.168.53.1")
        self.assertEqual(str(DroneIp.WIRELESS), "192.168.42.1")
        self.assertEqual(str(DroneIp.SIMULATED), "10.202.0.1")

    def test_enum_members(self):
        expected_values = {"192.168.53.1", "192.168.42.1", "10.202.0.1"}
        enum_values = {member.value for member in DroneIp}
        self.assertEqual(enum_values, expected_values)


if __name__ == "__main__":
    unittest.main()
