import unittest

from drone_base.control.manual.operations import MovementVector


class TestMovementVector(unittest.TestCase):
    """Tests for the MovementVector dataclass"""

    def test_default_initialization(self):
        """Test default initialization with zeros"""
        movement = MovementVector()
        self.assertEqual(movement.x, 0)
        self.assertEqual(movement.y, 0)
        self.assertEqual(movement.z, 0)
        self.assertEqual(movement.z_rot, 0)

    def test_custom_initialization(self):
        """Test initialization with custom values"""
        movement = MovementVector(x=10, y=-5, z=3, z_rot=45)
        self.assertEqual(movement.x, 10)
        self.assertEqual(movement.y, -5)
        self.assertEqual(movement.z, 3)
        self.assertEqual(movement.z_rot, 45)

        # Test partial initialization
        movement = MovementVector(x=10, z=3)
        self.assertEqual(movement.x, 10)
        self.assertEqual(movement.y, 0)  # Default
        self.assertEqual(movement.z, 3)
        self.assertEqual(movement.z_rot, 0)  # Default

    def test_reset_method(self):
        """Test the reset method sets all values to zero"""
        movement = MovementVector(x=10, y=-5, z=3, z_rot=45)
        movement.reset()
        self.assertEqual(movement.x, 0)
        self.assertEqual(movement.y, 0)
        self.assertEqual(movement.z, 0)
        self.assertEqual(movement.z_rot, 0)

    def test_is_moving_method(self):
        """Test the is_moving method correctly detects movement"""
        movement = MovementVector()
        self.assertFalse(movement.is_moving())

        test_cases = [
            {'x': 1, 'expected': True},
            {'y': -1, 'expected': True},
            {'z': 2, 'expected': True},
            {'z_rot': 90, 'expected': True},
        ]

        for case in test_cases:
            with self.subTest(**case):
                kwargs = {k: v for k, v in case.items() if k != 'expected'}
                movement = MovementVector(**kwargs)
                self.assertEqual(movement.is_moving(), case['expected'])

        movement = MovementVector(x=0, y=-5, z=0, z_rot=45)
        self.assertTrue(movement.is_moving())

        movement.reset()
        self.assertFalse(movement.is_moving())

    def test_immutability(self):
        """Test that dataclass instances are mutable as expected"""
        movement = MovementVector(x=10)
        movement.x = 20
        self.assertEqual(movement.x, 20)

    def test_equality(self):
        """Test that dataclass provides equality comparison"""
        movement1 = MovementVector(x=10, y=-5, z=3, z_rot=45)
        movement2 = MovementVector(x=10, y=-5, z=3, z_rot=45)
        movement3 = MovementVector(x=10, y=-5, z=3, z_rot=0)

        self.assertEqual(movement1, movement2)
        self.assertNotEqual(movement1, movement3)


if __name__ == "__main__":
    unittest.main()
