import unittest

from drone_base.control.states import DroneState, FlightState, GimbalOrientation


class TestDroneState(unittest.TestCase):
    def setUp(self):
        """Set up a fresh DroneState instance before each test."""
        self.drone_state = DroneState()

    def test_initial_state(self):
        """Test that the drone initializes with correct default values."""
        self.assertEqual(self.drone_state.flight_state, FlightState.INITIAL)
        self.assertEqual(self.drone_state.gimbal_orientation, GimbalOrientation.HORIZONTAL)

    def test_set_flight_state(self):
        """Test setting flight state works correctly."""
        self.drone_state.set_flight_state(FlightState.FLYING)
        self.assertEqual(self.drone_state.flight_state, FlightState.FLYING)

        self.drone_state.set_flight_state(FlightState.LANDED)
        self.assertEqual(self.drone_state.flight_state, FlightState.LANDED)

    def test_set_gimbal_orientation(self):
        """Test setting gimbal orientation works correctly."""
        self.drone_state.set_gimbal_orientation(GimbalOrientation.TILTED)
        self.assertEqual(self.drone_state.gimbal_orientation, GimbalOrientation.TILTED)

        self.drone_state.set_gimbal_orientation(GimbalOrientation.DOWNWARD)
        self.assertEqual(self.drone_state.gimbal_orientation, GimbalOrientation.DOWNWARD)

    def test_get_state(self):
        """Test that get_state returns the correct tuple of states."""
        self.drone_state.flight_state = FlightState.FLYING
        self.drone_state.gimbal_orientation = GimbalOrientation.TILTED

        state_tuple = self.drone_state.get_state()
        self.assertEqual(state_tuple, (FlightState.FLYING, GimbalOrientation.TILTED))

    def test_get_orientation_on_pitch_horizontal(self):
        """Test pitch value of 0 returns HORIZONTAL orientation."""
        orientation = self.drone_state.get_orientation_on_pitch(0)
        self.assertEqual(orientation, GimbalOrientation.HORIZONTAL)

    def test_get_orientation_on_pitch_tilted(self):
        """Test pitch values between 0 and 90 return TILTED orientation."""
        test_values = [1, 15, 30, 45, 89]
        for value in test_values:
            orientation = self.drone_state.get_orientation_on_pitch(value)
            self.assertEqual(orientation, GimbalOrientation.TILTED)

            orientation = self.drone_state.get_orientation_on_pitch(-value)
            self.assertEqual(orientation, GimbalOrientation.TILTED)

    def test_get_orientation_on_pitch_downward(self):
        """Test pitch value of 90 or greater returns DOWNWARD orientation."""
        test_values = [90, 100, 180]
        for value in test_values:
            orientation = self.drone_state.get_orientation_on_pitch(value)
            self.assertEqual(orientation, GimbalOrientation.DOWNWARD)

            orientation = self.drone_state.get_orientation_on_pitch(-value)
            self.assertEqual(orientation, GimbalOrientation.DOWNWARD)

    def test_flight_state_string_representation(self):
        """Test that FlightState string representation works correctly."""
        self.assertEqual(str(FlightState.INITIAL), "initial-landed")
        self.assertEqual(str(FlightState.LANDED), "landed")
        self.assertEqual(str(FlightState.FLYING), "flying")


if __name__ == '__main__':
    unittest.main()
