#!/usr/bin/env python
# Copyright (c) Meta Platforms, Inc. and affiliates.

# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.


import argparse
import builtins
import os
import random
import shutil
import time
import warnings
import json

import torch
import torch.backends.cudnn as cudnn
import torch.distributed as dist
import torch.multiprocessing as mp
import torch.nn as nn
import torch.nn.parallel
import torch.optim
import torch.utils.data
import torch.utils.data.distributed
import torchvision
import torchvision.datasets as datasets
import torchvision.models as models
import torchvision.transforms as transforms
from rotated_image_folder import RotatedImageFolder
import numpy as np

from moco.builder import PredictionMLP
from hyper.models import InvertibleNet, FiLMAlign, CondAlignMLP, MultiLinearAlign, WarpAlign
from hyper.hypernetwork import EquiRotate


model_names = sorted(
    name
    for name in models.__dict__
    if name.islower() and not name.startswith("__") and callable(models.__dict__[name])
)

parser = argparse.ArgumentParser(description="PyTorch ImageNet Training")
parser.add_argument("data", metavar="DIR", help="path to dataset")
parser.add_argument(
    "-a",
    "--arch",
    metavar="ARCH",
    default="resnet50",
    choices=model_names,
    help="model architecture: " + " | ".join(model_names) + " (default: resnet50)",
)
parser.add_argument(
    "-j",
    "--workers",
    default=32,
    type=int,
    metavar="N",
    help="number of data loading workers (default: 32)",
)
parser.add_argument(
    "--epochs", default=100, type=int, metavar="N", help="number of total epochs to run"
)
parser.add_argument(
    "--start-epoch",
    default=0,
    type=int,
    metavar="N",
    help="manual epoch number (useful on restarts)",
)
parser.add_argument(
    "-b",
    "--batch-size",
    default=256,
    type=int,
    metavar="N",
    help="mini-batch size (default: 256), this is the total "
    "batch size of all GPUs on the current node when "
    "using Data Parallel or Distributed Data Parallel",
)
parser.add_argument(
    "--lr",
    "--learning-rate",
    default=30.0,
    type=float,
    metavar="LR",
    help="initial learning rate",
    dest="lr",
)
parser.add_argument(
    "--schedule",
    default=[60, 80],
    nargs="*",
    type=int,
    help="learning rate schedule (when to drop lr by a ratio)",
)
parser.add_argument("--momentum", default=0.9, type=float, metavar="M", help="momentum")
parser.add_argument(
    "--wd",
    "--weight-decay",
    default=0.0,
    type=float,
    metavar="W",
    help="weight decay (default: 0.)",
    dest="weight_decay",
)
parser.add_argument(
    "-p",
    "--print-freq",
    default=10,
    type=int,
    metavar="N",
    help="print frequency (default: 10)",
)
parser.add_argument(
    "--resume",
    default="",
    type=str,
    metavar="PATH",
    help="path to latest checkpoint (default: none)",
)
parser.add_argument(
    "-e",
    "--evaluate",
    dest="evaluate",
    action="store_true",
    help="evaluate model on validation set",
)
parser.add_argument(
    "--world-size",
    default=-1,
    type=int,
    help="number of nodes for distributed training",
)
parser.add_argument(
    "--rank", default=-1, type=int, help="node rank for distributed training"
)
parser.add_argument(
    "--dist-url",
    default="tcp://224.66.41.62:23456",
    type=str,
    help="url used to set up distributed training",
)
parser.add_argument(
    "--dist-backend", default="nccl", type=str, help="distributed backend"
)
parser.add_argument(
    "--seed", default=None, type=int, help="seed for initializing training. "
)
parser.add_argument("--gpu", default=None, type=int, help="GPU id to use.")
parser.add_argument(
    "--multiprocessing-distributed",
    action="store_true",
    help="Use multi-processing distributed training to launch "
    "N processes per node, which has N GPUs. This is the "
    "fastest way to use PyTorch for either single node or "
    "multi node data parallel training",
)

parser.add_argument(
    "--pretrained", default="checkpoint_0399.pth.tar", type=str, help="path to moco pretrained checkpoint"
)
parser.add_argument('--pretrained_dir', default='./experiments/stl10_mocov2_resnet18')
parser.add_argument('--imsize', default=96, type=int)
parser.add_argument('--pretrain-set', default='stl10', type=str, help='pretrain dataset',
                    choices=['stl10','stl10-R','imagenet100','imagenet100-R','imagenet1k','caltech256','mtarsi','imagenet100_app', 'EMNIST', 'EMNIST-R'])
parser.add_argument('--eval-set', default='stl10', type=str, help='evaluation dataset',
                    choices=['stl10','imagenet100','stanford_cars','fgvc_aircraft','cub_200_2011','cifar10','cifar100','caltech256','imagenet1k','mtarsi','oxford_flowers', 'EMNIST', 'RotNIST'])
parser.add_argument('--gie', default=False, action='store_true')
parser.add_argument('--connector', default='softmax', type=str)
parser.add_argument('--num_classes', default=10, type=int)
parser.add_argument('--rotated', default=False, action='store_true')
parser.add_argument('--pred_hidden_dim', default=512, type=int)
parser.add_argument('--R_choices', default='linear', type=str,
                    choices=['linear', 'film', 'cond_align', 'multi_linear_align', 'affine_coupling', 'warp_align'])
parser.add_argument('--eqv_loss_type', default='mse', type=str, choices=['mse', 'infonce', 'shift'])
parser.add_argument('--deep', default=False, action='store_true')
parser.add_argument('--depth', default=3, type=int)
parser.add_argument('--use_mlp', default=False, action='store_true')
parser.add_argument('--scale_embedding', default=False, action='store_true')
parser.add_argument('--multi_num', default=3, type=int)
parser.add_argument('--reverse', default=False, action='store_true')
parser.add_argument('--stop_gradient1', default=False, action='store_true')
parser.add_argument('--stop_gradient2', default=False, action='store_true')
parser.add_argument('--iterations', default=0, type=int)

best_acc1 = 0


def main() -> None:
    args = parser.parse_args()

    seed = np.random.randint(0, 10000)
    args.seed = seed
    if args.seed is not None:
        random.seed(args.seed)
        torch.manual_seed(args.seed)
        cudnn.deterministic = True
        warnings.warn(
            "You have chosen to seed training. "
            "This will turn on the CUDNN deterministic setting, "
            "which can slow down your training considerably! "
            "You may see unexpected behavior when restarting "
            "from checkpoints."
        )

    if args.gpu is not None:
        warnings.warn(
            "You have chosen a specific GPU. This will completely "
            "disable data parallelism."
        )

    if args.dist_url == "env://" and args.world_size == -1:
        args.world_size = int(os.environ["WORLD_SIZE"])

    args.distributed = args.world_size > 1 or args.multiprocessing_distributed

    ngpus_per_node = torch.cuda.device_count()
    if args.multiprocessing_distributed:
        # Since we have ngpus_per_node processes per node, the total world_size
        # needs to be adjusted accordingly
        args.world_size = ngpus_per_node * args.world_size
        # Use torch.multiprocessing.spawn to launch distributed processes: the
        # main_worker process function
        mp.spawn(main_worker, nprocs=ngpus_per_node, args=(ngpus_per_node, args))
    else:
        # Simply call main_worker function
        main_worker(args.gpu, ngpus_per_node, args)


def main_worker(gpu, ngpus_per_node, args) -> None:
    global best_acc1
    args.gpu = gpu
    
    print('Seed Num: {}'.format(args.seed))
    stats = {
        'Seed': args.seed
    }
    save_stats(stats, args.pretrained_dir, args)
    # suppress printing if not master
    if args.multiprocessing_distributed and args.gpu != 0:

        def print_pass(*args) -> None:
            pass

        builtins.print = print_pass

    if args.gpu is not None:
        # pyre-fixme[61]: `print` is undefined, or not always defined.
        print("Use GPU: {} for training".format(args.gpu))

    if args.distributed:
        if args.dist_url == "env://" and args.rank == -1:
            args.rank = int(os.environ["RANK"])
        if args.multiprocessing_distributed:
            # For multiprocessing distributed training, rank needs to be the
            # global rank among all the processes
            args.rank = args.rank * ngpus_per_node + gpu
        dist.init_process_group(
            backend=args.dist_backend,
            init_method=args.dist_url,
            world_size=args.world_size,
            rank=args.rank,
        )
    # create model
    # pyre-fixme[61]: `print` is undefined, or not always defined.
    print("=> creating model '{}'".format(args.arch))
    model = Encoder(args)
    # model = models.__dict__[args.arch]()

    # freeze all layers but the last fc
    for name, param in model.named_parameters():
        if name not in ["fc.weight", "fc.bias"]:
            param.requires_grad = False
    # init the fc layer
    model.fc.weight.data.normal_(mean=0.0, std=0.01)
    model.fc.bias.data.zero_()
    
    print(model)

    # load from pre-trained, before DistributedDataParallel constructor
    if args.pretrained:
        if os.path.isfile(os.path.join(args.pretrained_dir,args.pretrained)):
            # pyre-fixme[61]: `print` is undefined, or not always defined.
            print("=> loading checkpoint '{}'".format(os.path.join(args.pretrained_dir,args.pretrained)))
            checkpoint = torch.load(os.path.join(args.pretrained_dir,args.pretrained), map_location="cpu")

            # rename moco pre-trained keys
            state_dict = checkpoint["state_dict"]
            encoder_q_state_dict = {k.replace("module.encoder_q.", ""): v for k, v in state_dict.items() if k.startswith("module.encoder_q")}
                
            args.start_epoch = 0
            msg = model.backbone.load_state_dict(encoder_q_state_dict, strict=False)
            print(msg.missing_keys)
            print(msg.unexpected_keys)   
            
            if args.gie:
                if args.stop_gradient1:
                    predictor_q_state_dict = {k.replace("module.predictor_q.", ""): v for k, v in state_dict.items() if k.startswith("module.predictor_q")}
                    msg = model.predictor.load_state_dict(predictor_q_state_dict, strict=False)
                    print(msg.missing_keys)
                    print(msg.unexpected_keys)  
                    
                elif args.stop_gradient2:
                    predictor_state_dict = {k.replace("module.predictor.", ""): v for k, v in state_dict.items() if k.startswith("module.predictor")}
                    msg = model.predictor.load_state_dict(predictor_state_dict, strict=False)
                    print(msg.missing_keys)
                    print(msg.unexpected_keys)      
                
                equi_transform_state_dict = {k.replace("module.equi_transform.", ""): v for k, v in state_dict.items() if k.startswith("module.equi_transform")}
                msg = model.equi_transform.load_state_dict(equi_transform_state_dict, strict=False)
                print(msg.missing_keys)
                print(msg.unexpected_keys)
                
        else:
            # pyre-fixme[61]: `print` is undefined, or not always defined.
            print("=> no checkpoint found at '{}'".format(args.pretrained))

    if args.distributed:
        # For multiprocessing distributed, DistributedDataParallel constructor
        # should always set the single device scope, otherwise,
        # DistributedDataParallel will use all available devices.
        if args.gpu is not None:
            torch.cuda.set_device(args.gpu)
            model.cuda(args.gpu)
            # When using a single GPU per process and per
            # DistributedDataParallel, we need to divide the batch size
            # ourselves based on the total number of GPUs we have
            args.batch_size = int(args.batch_size / ngpus_per_node)
            args.workers = int((args.workers + ngpus_per_node - 1) / ngpus_per_node)
            model = torch.nn.parallel.DistributedDataParallel(
                model, device_ids=[args.gpu]
            )
        else:
            model.cuda()
            # DistributedDataParallel will divide and allocate batch_size to all
            # available GPUs if device_ids are not set
            model = torch.nn.parallel.DistributedDataParallel(model)
    elif args.gpu is not None:
        torch.cuda.set_device(args.gpu)
        model = model.cuda(args.gpu)
    else:
        # DataParallel will divide and allocate batch_size to all available GPUs
        if args.arch.startswith("alexnet") or args.arch.startswith("vgg"):
            model.features = torch.nn.DataParallel(model.features)
            model.cuda()
        else:
            model = torch.nn.DataParallel(model).cuda()

    # define loss function (criterion) and optimizer
    criterion = nn.CrossEntropyLoss().cuda(args.gpu)

    # optimize only the linear classifier
    parameters = list(filter(lambda p: p.requires_grad, model.parameters()))
    assert len(parameters) == 2  # fc.weight, fc.bias
    optimizer = torch.optim.SGD(
        parameters, args.lr, momentum=args.momentum, weight_decay=args.weight_decay
    )

    # optionally resume from a checkpoint
    if args.resume:
        if os.path.isfile(args.resume):
            # pyre-fixme[61]: `print` is undefined, or not always defined.
            print("=> loading checkpoint '{}'".format(args.resume))
            if args.gpu is None:
                checkpoint = torch.load(args.resume)
            else:
                # Map model to be loaded to specified single gpu.
                loc = "cuda:{}".format(args.gpu)
                checkpoint = torch.load(args.resume, map_location=loc)
            args.start_epoch = checkpoint["epoch"]
            best_acc1 = checkpoint["best_acc1"]
            if args.gpu is not None:
                # best_acc1 may be from a checkpoint from a different GPU
                best_acc1 = best_acc1.to(args.gpu)
            model.load_state_dict(checkpoint["state_dict"])
            optimizer.load_state_dict(checkpoint["optimizer"])
            # pyre-fixme[61]: `print` is undefined, or not always defined.
            print(
                "=> loaded checkpoint '{}' (epoch {})".format(
                    args.resume, checkpoint["epoch"]
                )
            )
        else:
            # pyre-fixme[61]: `print` is undefined, or not always defined.
            print("=> no checkpoint found at '{}'".format(args.resume))

    cudnn.benchmark = True

    # Data loading code
    if args.pretrain_set == 'imagenet' or args.pretrain_set == 'imagenet100-R':
        normalize = transforms.Normalize(
            mean=[0.485, 0.456, 0.406], std=[0.229, 0.224, 0.225]
        )
        traindir = os.path.join(args.data, "train")
        valdir = os.path.join(args.data, "val")
        
    elif args.pretrain_set == 'stl10' or args.pretrain_set =='stl10-R':
        normalize = transforms.Normalize(
            mean=[0.43, 0.42, 0.39], std=[0.27, 0.26, 0.27]
        )
        traindir = os.path.join(args.data, "train")
        valdir = os.path.join(args.data, "test")

    if not args.rotated:
        train_dataset = datasets.ImageFolder(
            traindir,
            transforms.Compose(
                [
                    transforms.RandomResizedCrop(args.imsize),
                    transforms.RandomHorizontalFlip(),
                    transforms.ToTensor(),
                    normalize,
                ]
            ),
        )
    else:
        train_dataset = RotatedImageFolder(
            traindir,
            N=4,
            transform=[transforms.Compose(
                [
                    transforms.RandomResizedCrop(args.imsize),
                    transforms.RandomHorizontalFlip(),
                    transforms.ToTensor(),
                ]
            ), normalize], 
        )

    if args.distributed:
        train_sampler = torch.utils.data.distributed.DistributedSampler(train_dataset)
    else:
        train_sampler = None

    train_loader = torch.utils.data.DataLoader(
        train_dataset,
        batch_size=args.batch_size,
        shuffle=(train_sampler is None),
        num_workers=args.workers,
        pin_memory=True,
        sampler=train_sampler,
    )

    if args.pretrain_set == 'imagenet100' or args.pretrain_set == 'imagenet100-R':
        if not args.rotated:
            val_loader = torch.utils.data.DataLoader(
                datasets.ImageFolder(
                    valdir,
                    transforms.Compose(
                        [
                            transforms.Resize(256),
                            transforms.CenterCrop(224),
                            transforms.ToTensor(),
                            normalize,
                        ]
                    ),
                ),
                batch_size=args.batch_size,
                shuffle=False,
                num_workers=args.workers,
                pin_memory=True,
            )
        else:
            val_loader = torch.utils.data.DataLoader(
            RotatedImageFolder(
                valdir,
                N=4,
                transform=[transforms.Compose(
                    [
                        transforms.Resize(256),
                        transforms.CenterCrop(224),
                        transforms.ToTensor(),
                    ]
                ), normalize]
            ),
            batch_size=args.batch_size,
            shuffle=False,
            num_workers=args.workers,
            pin_memory=True,
        )
            
    elif args.pretrain_set == 'stl10' or args.pretrain_set == 'stl10-R':
        if not args.rotated:
            val_loader = torch.utils.data.DataLoader(
                datasets.ImageFolder(
                    valdir,
                    transforms.Compose(
                        [
                            transforms.Resize(args.imsize),
                            transforms.CenterCrop(args.imsize),
                            transforms.ToTensor(),
                            normalize,
                        ]
                    ),
                ),
                batch_size=args.batch_size,
                shuffle=False,
                num_workers=args.workers,
                pin_memory=True,
            )
        else:
            val_loader = torch.utils.data.DataLoader(
            RotatedImageFolder(
                valdir,
                N=4,
                transform=[transforms.Compose(
                    [
                        transforms.Resize(args.imsize),
                        transforms.CenterCrop(args.imsize),
                        transforms.ToTensor(),
                    ]
                ), normalize]
            ),
            batch_size=args.batch_size,
            shuffle=False,
            num_workers=args.workers,
            pin_memory=True,
        )

    if args.evaluate:
        validate(val_loader, model, criterion, args)
        return

    for epoch in range(args.start_epoch, args.epochs):
        if args.distributed:
            # pyre-fixme[16]: Optional type has no attribute `set_epoch`.
            train_sampler.set_epoch(epoch)
        lr = adjust_learning_rate(optimizer, epoch, args)

        # train for one epoch
        train(train_loader, model, criterion, optimizer, epoch, args, lr)

        # evaluate on validation set
        acc1 = validate(val_loader, model, criterion, args)

        # remember best acc@1 and save checkpoint
        is_best = acc1 > best_acc1
        best_acc1 = max(acc1, best_acc1)

        stats = {
            "epoch": epoch,
            "acc@1": acc1.item(),
            "best acc@1": best_acc1.item()
        }
        save_stats(stats, args.pretrained_dir, args)
        
        if not args.multiprocessing_distributed or (
            args.multiprocessing_distributed and args.rank % ngpus_per_node == 0
        ):
            save_checkpoint(
                {
                    "epoch": epoch + 1,
                    "arch": args.arch,
                    "state_dict": model.state_dict(),
                    "best_acc1": best_acc1,
                    "optimizer": optimizer.state_dict(),
                },
                is_best,
                args.pretrained_dir,
                args
            )
            if epoch == args.start_epoch:
                sanity_check(model.state_dict(), args.pretrained_dir, args.pretrained, args)


def train(train_loader, model, criterion, optimizer, epoch, args, lr) -> None:
    batch_time = AverageMeter("Time", ":6.3f")
    data_time = AverageMeter("Data", ":6.3f")
    losses = AverageMeter("Loss", ":.4e")
    top1 = AverageMeter("Acc@1", ":6.2f")
    top5 = AverageMeter("Acc@5", ":6.2f")
    progress = ProgressMeter(
        len(train_loader),
        [batch_time, data_time, losses, top1, top5],
        prefix="Epoch: [{}]".format(epoch),
    )

    """
    Switch to eval mode:
    Under the protocol of linear classification on frozen features/models,
    it is not legitimate to change any part of the pre-trained model.
    BatchNorm in train mode may revise running mean/std (even if it receives
    no gradient), which are part of the model parameters too.
    """
    model.eval()

    end = time.time()
    for step, (images, target) in enumerate(train_loader):
        # measure data loading time
        data_time.update(time.time() - end)

        if args.gpu is not None:
            images = images.cuda(args.gpu, non_blocking=True)
        target = target.cuda(args.gpu, non_blocking=True)

        # compute output
        output = model(images)
        loss = criterion(output, target)

        # measure accuracy and record loss
        acc1, acc5 = accuracy(output, target, topk=(1, 5))
        losses.update(loss.item(), images.size(0))
        top1.update(acc1[0], images.size(0))
        top5.update(acc5[0], images.size(0))

        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        # measure elapsed time
        batch_time.update(time.time() - end)
        end = time.time()

        if step % args.print_freq == 0:
            progress.display(step)
        if step % 10 == 0 or step == len(train_loader) - 1:
            stats = {
                "epoch": epoch,
                "step": step,
                "loss": losses.avg,
                "acc@1": top1.avg.item(),
                "acc@5": top5.avg.item(),
                "lr": lr,
            }
            save_stats(stats, args.pretrained_dir, args)

def validate(val_loader, model, criterion, args):
    batch_time = AverageMeter("Time", ":6.3f")
    losses = AverageMeter("Loss", ":.4e")
    top1 = AverageMeter("Acc@1", ":6.2f")
    top5 = AverageMeter("Acc@5", ":6.2f")
    progress = ProgressMeter(
        len(val_loader), [batch_time, losses, top1, top5], prefix="Test: "
    )

    # switch to evaluate mode
    model.eval()

    with torch.no_grad():
        end = time.time()
        for i, (images, target) in enumerate(val_loader):
            if args.gpu is not None:
                images = images.cuda(args.gpu, non_blocking=True)
            target = target.cuda(args.gpu, non_blocking=True)

            # compute output
            output = model(images)
            loss = criterion(output, target)

            # measure accuracy and record loss
            acc1, acc5 = accuracy(output, target, topk=(1, 5))
            losses.update(loss.item(), images.size(0))
            top1.update(acc1[0], images.size(0))
            top5.update(acc5[0], images.size(0))

            # measure elapsed time
            batch_time.update(time.time() - end)
            end = time.time()

            if i % args.print_freq == 0:
                progress.display(i)

        # TODO: this should also be done with the ProgressMeter
        print(
            " * Acc@1 {top1.avg:.3f} Acc@5 {top5.avg:.3f}".format(top1=top1, top5=top5)
        )
        stats = {
            'Acc@1': top1.avg.item(),
            'Acc@5': top5.avg.item()
        }
        save_stats(stats, args.pretrained_dir, args)

    return top1.avg


def save_checkpoint(state, is_best, filepath, args) -> None:
    if args.gie:
        if args.rotated:
            checkpoint_path = os.path.join(filepath, f'checkpoint_inv_r_{args.iterations}.pth.tar')
            best_model_path = os.path.join(filepath, f'model_best_inv_r_{args.iterations}.pth.tar')
        else:
            checkpoint_path = os.path.join(filepath, f'checkpoint_inv_{args.iterations}.pth.tar')
            best_model_path = os.path.join(filepath, f'model_best_inv_{args.iterations}.pth.tar')
    else:
        if args.rotated:
            checkpoint_path = os.path.join(filepath, f'checkpoint_r_{args.iterations}.pth.tar')
            best_model_path = os.path.join(filepath, f'model_best_r_{args.iterations}.pth.tar')
        else:
            checkpoint_path = os.path.join(filepath, f'checkpoint_{args.iterations}.pth.tar')
            best_model_path = os.path.join(filepath, f'model_best_{args.iterations}.pth.tar')
    torch.save(state, checkpoint_path)
    if is_best:
        shutil.copyfile(checkpoint_path, best_model_path)

def save_stats(stats, checkpoint_dir, args):
    # Only execute on main process (rank 0)
    if dist.is_initialized() and dist.get_rank() != 0:
        return

    # Ensure directory exists
    if not os.path.exists(checkpoint_dir):
        os.makedirs(checkpoint_dir)
    """Save stats to a text file."""
    if args.gie:
        if args.rotated:
            stats_file = os.path.join(checkpoint_dir, f"lincls_stats_inv_rotated_{args.iterations}.txt")
        else:
            stats_file = os.path.join(checkpoint_dir, f"lincls_stats_inv_{args.iterations}.txt")
    else:
        if args.rotated:
            stats_file = os.path.join(checkpoint_dir, f"lincls_stats_rotated_{args.iterations}.txt")
        else:
            stats_file = os.path.join(checkpoint_dir, f"lincls_stats_{args.iterations}.txt")
    with open(stats_file, "a") as f:
        json.dump(stats, f)
        f.write("\n")
        
def sanity_check(state_dict, pretrained_dir, pretrained_weights, args) -> None:
    """
    Linear classifier should not change any weights other than the linear layer.
    This sanity check asserts nothing wrong happens (e.g., BN stats updated).
    """
    print("=> loading '{}' for sanity check".format(pretrained_weights))
    checkpoint = torch.load(os.path.join(pretrained_dir, pretrained_weights), map_location="cpu")
    state_dict_pre = checkpoint["state_dict"]

    for k in list(state_dict.keys()):
        # only ignore fc layer
        if "fc.weight" in k or "fc.bias" in k:
            continue

        # name in pretrained model
        if args.stop_gradient1:
            k_pre = (
                "module.encoder_q." + k[len("backbone.") :] if k.startswith("backbone.") else
                "module.encoder_q." + k[len("module.backbone.") :] if k.startswith("module.backbone.") else
                "module.predictor_q." + k[len("module.predictor.") :] if k.startswith("module.predictor.") else
                "module.equi_transform." + k[len("module.equi_transform.") :] if k.startswith("module.equi_transform.") else
                "module.encoder_q." + k[len("module.") :] if k.startswith("module.") else
                "module.encoder_q." + k
            )
        elif args.stop_gradient2:
            k_pre = (
                "module.encoder_q." + k[len("backbone.") :] if k.startswith("backbone.")
                else "module.encoder_q." + k[len("module.backbone.") :] if k.startswith("module.backbone.")
                else "module.predictor." + k[len("module.predictor.") :] if k.startswith("module.predictor.")
                else "module.equi_transform." + k[len("module.equi_transform.") :] if k.startswith("module.equi_transform.")
                else "module.encoder_q." + k[len("module.") :] if k.startswith("module.")
                else "module.encoder_q." + k
            )
        else:
            k_pre = (
                "module.encoder_q." + k[len("backbone.") :] if k.startswith("backbone.")
                else "module.encoder_q." + k[len("module.backbone.") :] if k.startswith("module.backbone.")
                else "module.encoder_q." + k[len("module.") :] if k.startswith("module.")
                else "module.encoder_q." + k
            )

        assert (
            state_dict[k].cpu() == state_dict_pre[k_pre]
        ).all(), "{} is changed in linear classifier training.".format(k)

    print("=> sanity check passed.")


class AverageMeter:
    """Computes and stores the average and current value"""

    def __init__(self, name, fmt: str = ":f") -> None:
        self.name = name
        self.fmt = fmt
        self.reset()

    def reset(self) -> None:
        self.val = 0
        self.avg = 0
        self.sum = 0
        self.count = 0

    def update(self, val, n: int = 1) -> None:
        self.val = val
        self.sum += val * n
        self.count += n
        self.avg = self.sum / self.count

    def __str__(self) -> str:
        fmtstr = "{name} {val" + self.fmt + "} ({avg" + self.fmt + "})"
        return fmtstr.format(**self.__dict__)


class ProgressMeter:
    def __init__(self, num_batches, meters, prefix: str = "") -> None:
        self.batch_fmtstr = self._get_batch_fmtstr(num_batches)
        self.meters = meters
        self.prefix = prefix

    def display(self, batch) -> None:
        entries = [self.prefix + self.batch_fmtstr.format(batch)]
        entries += [str(meter) for meter in self.meters]
        print("\t".join(entries))

    def _get_batch_fmtstr(self, num_batches):
        num_digits = len(str(num_batches // 1))
        fmt = "{:" + str(num_digits) + "d}"
        return "[" + fmt + "/" + fmt.format(num_batches) + "]"


def adjust_learning_rate(optimizer, epoch, args) -> None:
    """Decay the learning rate based on schedule"""
    lr = args.lr
    for milestone in args.schedule:
        lr *= 0.1 if epoch >= milestone else 1.0
    for param_group in optimizer.param_groups:
        param_group["lr"] = lr
    return lr


def accuracy(output, target, topk=(1,)):
    """Computes the accuracy over the k top predictions for the specified values of k"""
    with torch.no_grad():
        maxk = max(topk)
        batch_size = target.size(0)

        _, pred = output.topk(maxk, 1, True, True)
        pred = pred.t()
        correct = pred.eq(target.view(1, -1).expand_as(pred))

        res = []
        for k in topk:
            # correct_k = correct[:k].view(-1).float().sum(0, keepdim=True)
            correct_k = correct[:k].reshape(-1).float().sum(0, keepdim=True)
            res.append(correct_k.mul_(100.0 / batch_size))
        return res

class Encoder(nn.Module):

    def __init__(self, args):
        super().__init__()

        self.order = 4
        self.gie = args.gie
        self.args= args

        if args.arch=='resnet18':
            self.backbone = torchvision.models.resnet18(zero_init_residual=True)
            self.backbone.fc = nn.Identity()
            feature_dim = 512

        elif args.arch=='resnet50':
            self.backbone = torchvision.models.resnet50(zero_init_residual=True)
            self.backbone.fc = nn.Identity()
            feature_dim = 2048

        if self.gie:
            self.predictor = PredictionMLP(feature_dim, args.pred_hidden_dim, 4)
            
            if not args.eqv_loss_type == 'shift':
                if args.R_choices == 'linear':
                    self.equi_transform = EquiRotate(feature_dim, self.args.use_mlp)
                elif args.R_choices == 'film':
                    self.equi_transform = FiLMAlign(args, feature_dim, feature_dim*2)
                elif args.R_choices == 'cond_align':
                    self.equi_transform = CondAlignMLP(args, feature_dim, embed_dim=16, hidden_dim=feature_dim*2)
                elif args.R_choices == 'multi_linear_align':
                    self.equi_transform = MultiLinearAlign(args, feature_dim)
                elif args.R_choices == 'affine_coupling':
                    self.equi_transform = InvertibleNet(args, feature_dim, feature_dim*2, 2)
                elif args.R_choices == 'warp_align':
                    self.equi_transform = WarpAlign(feature_dim, feature_dim*2)

        if args.connector=='softmax':
            self.connector = torch.nn.Softmax(dim=1)
        elif args.connector=='shift':
            self.connector = None
            permute_patterns = [torch.roll(torch.arange(self.order), shifts=-i).tolist() for i in range(self.order)]
            self.permute_tensor = torch.tensor(permute_patterns).cuda()


        self.fc = nn.Linear(feature_dim, args.num_classes)
    
    def forward(self, x):

        if not self.gie:
            x = self.backbone(x)
            out = self.fc(x)
        else:
            FX = self.backbone(x)
            eqv_logit = self.predictor(FX).flatten(1)
            b,c = FX.shape
            if self.connector:
                eqv_score = self.connector(eqv_logit)
                
                if self.args.eqv_loss_type == 'shift':
                    FX_re = FX.reshape([b, c//self.order, self.order])
                    permuted_reprs = [torch.roll(FX_re, shifts=-i, dims=2).reshape([b,c]) for i in range(self.order)]
                    permuted_reprs = torch.stack(permuted_reprs, dim=-1)
                    HX = torch.matmul(permuted_reprs, eqv_score.unsqueeze(dim=-1)).squeeze()
                else:
                    if self.args.R_choices == 'linear' or self.args.R_choices == 'affine_coupling' or self.args.R_choices == 'warp_align':
                        permuted_reprs = [FX]
                        for _ in range(self.order-1):
                            permuted_reprs.append(self.equi_transform(permuted_reprs[-1]))

                        order = [0, 3, 2, 1]
                        permuted_reprs = [permuted_reprs[i] for i in order]
                        permuted_reprs = torch.stack(permuted_reprs, dim=-1)
                        HX = torch.matmul(permuted_reprs, eqv_score.unsqueeze(dim=-1)).squeeze()
                    
                    elif self.args.R_choices == 'film' or self.args.R_choices == 'cond_align' or self.args.R_choices == 'multi_linear_align':
                        if self.args.multi_num == 3:
                            permuted_reprs = [FX]
                            
                            order = [
                                torch.full((FX.size()[0],), 3, dtype=torch.long, device=FX.device),
                                torch.full((FX.size()[0],), 2, dtype=torch.long, device=FX.device),
                                torch.full((FX.size()[0],), 1, dtype=torch.long, device=FX.device)
                            ]
                            
                            for i in range(self.order-1):
                                permuted_reprs.append(self.equi_transform(FX, order[i]))
                            permuted_reprs = torch.stack(permuted_reprs, dim=-1)
                            HX = torch.matmul(permuted_reprs, eqv_score.unsqueeze(dim=-1)).squeeze()
                        else:
                            permuted_reprs = []
                            
                            order = [
                                torch.full((FX.size()[0],), 0, dtype=torch.long, device=FX.device),
                                torch.full((FX.size()[0],), 3, dtype=torch.long, device=FX.device),
                                torch.full((FX.size()[0],), 2, dtype=torch.long, device=FX.device),
                                torch.full((FX.size()[0],), 1, dtype=torch.long, device=FX.device)
                            ]
                            
                            for i in range(self.order):
                                permuted_reprs.append(self.equi_transform(FX, order[i]))
                            permuted_reprs = torch.stack(permuted_reprs, dim=-1)
                            HX = torch.matmul(permuted_reprs, eqv_score.unsqueeze(dim=-1)).squeeze()

            else:
                eqv_idx = torch.argmax(eqv_logit, dim=1)
                
                if self.args.eqv_loss_type == 'shift':    
                    batch_perm = self.permute_tensor[eqv_idx].unsqueeze(1).expand(-1,c//self.order,-1)
                    FX_re = FX.reshape([b, c//self.order, self.order])
                    FX_re = FX_re.gather(2, batch_perm)
                    HX = FX_re.reshape([b,c])
                else:
                    trans = (4 - eqv_idx) % self.order
                
                    if self.args.R_choices == 'linear' or self.args.R_choices == 'affine_coupling' or self.args.R_choices == 'warp_align':  
                        FX_all = [FX]
                        for _ in range(self.order-1):
                            FX_all.append(self.equi_transform(FX_all[-1]))
                        FX_stack = torch.stack(FX_all, dim=1)
                        trans = trans.view(-1, 1, 1).expand(-1, -1, FX.size()[-1])
                        HX = torch.gather(FX_stack, dim=1, index=trans).squeeze(1)
                    
                    elif self.args.R_choices == 'film' or self.args.R_choices == 'cond_align' or self.args.R_choices == 'multi_linear_align':
                        trans_idx = torch.full((FX.size()[0],), trans, dtype=torch.long, device=FX.device)
                        HX = self.equi_transform(FX, trans_idx)
                            
            out = self.fc(HX)

        return out
    
if __name__ == "__main__":
    main()
