# Copyright (c) Meta Platforms, Inc. and affiliates.

# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

import random

from PIL import ImageFilter


class TwoCropsTransform:
    """Take two random crops of one image as the query and key."""

    def __init__(self, args, base_transform, rot_transform) -> None:
        self.args = args
        self.base_transform = base_transform
        self.rot_transform = rot_transform

    def __call__(self, x):
        if self.args.gie:
            if not self.args.horizontal_flip_first:
                q, r1 = self.rot_transform(x)
                k, r2 = self.rot_transform(x)
                
                q = self.base_transform(q)
                k = self.base_transform(k)
            else:
                q = self.base_transform(x)
                k = self.base_transform(x)
                
                q, r1 = self.rot_transform(q)
                k, r2 = self.rot_transform(k)

            return [q, k, r1, r2]
        else:
            q = self.base_transform(x)
            k = self.base_transform(x)
            r1 = 0
            r2 = 0
            return [q, k, r1, r2]


class GaussianBlur:
    """Gaussian blur augmentation in SimCLR https://arxiv.org/abs/2002.05709"""

    def __init__(self, sigma=[0.1, 2.0]) -> None:
        self.sigma = sigma

    def __call__(self, x):
        sigma = random.uniform(self.sigma[0], self.sigma[1])
        x = x.filter(ImageFilter.GaussianBlur(radius=sigma))
        return x
