# Standard library imports
import os
import random
import math
from pathlib import Path
from multiprocessing import Pool

# Third-party imports
import pygame
import cv2
import numpy as np
from tqdm import tqdm

def make_sequence(index):
    # Initialize Pygame
    pygame.init()

    # Set window dimensions
    WINDOW_WIDTH = 800
    WINDOW_HEIGHT = 600
    screen = pygame.Surface((WINDOW_WIDTH, WINDOW_HEIGHT))  # Use Surface instead of actual window

    # Color definitions
    WHITE = (255, 255, 255)
    COLORS = [
        (200, 100, 100),  # Red building
        (100, 200, 100),  # Green building
        (100, 100, 200),  # Blue building
        (200, 200, 100),  # Yellow building
        (200, 100, 200),  # Pink-purple building
        (100, 200, 200),  # Cyan building
        (150, 75, 0),     # Brown building
        (160, 160, 160),  # Gray building
        (255, 160, 122),  # Light coral building
        (176, 196, 222),  # Light steel blue building
        (152, 251, 152),  # Pale green building
        (238, 130, 238),  # Violet building
        (255, 218, 185),  # Peach building
        (175, 238, 238),  # Pale cyan building
        (219, 112, 147),  # Pale violet red building
    ]

    # Character settings
    PLAYER_WIDTH = 40
    PLAYER_HEIGHT = 40
    player_x = WINDOW_WIDTH // 2  # Represents actual world coordinates
    camera_offset = 0
    player_facing_right = True
    PLAYER_SPEED = 30
    building_count = random.randint(0, 100)

    class Building:
        def __init__(self, x):
            self.width = random.randint(50, 150)
            self.max_height = random.randint(100, 400)
            self.current_height = self.max_height
            self.height = self.max_height
            self.x = x
            self.color = random.choice(COLORS)
            self.is_empty = random.random() < 0.3  # 30% chance of being empty
            self.growing = False
            self.grow_speed = 200

        def update(self):
            if self.growing and self.current_height < self.max_height:
                self.current_height += self.grow_speed
                self.height = self.current_height

    # Generate initial buildings (ensure empty plots are not adjacent)
    buildings = []
    x = -WINDOW_WIDTH * 10
    last_was_empty = False

    while x < WINDOW_WIDTH * 10:
        building = Building(x)
        if last_was_empty:
            building.is_empty = False

        if building.is_empty:
            building.max_height = 200

        last_was_empty = building.is_empty
        buildings.append(building)
        x += building.width

    clock = pygame.time.Clock()
    running = True

    # Generate action sequence
    actions = []
    MAX_LENGTH = 64
    possible_actions = ['LEFT', 'RIGHT'] * 3 + ['NONE']
    
    if random.random() < 0.6:
        while len(actions) < MAX_LENGTH:
            action = random.choice(possible_actions)
            action_length = random.randint(1, 8)
            action_length = min(action_length, MAX_LENGTH - len(actions))
            actions.extend([action] * action_length)
    else:
        if random.random() < 0.5:
            actions = ['LEFT'] * 32 + ['RIGHT'] * 32
        else:
            actions = ['RIGHT'] * 32 + ['LEFT'] * 32

    # Main game loop
    frame_count = 0
    while running:
        # Move character based on current action
        current_action = actions[min(frame_count, len(actions) - 1)]
        if current_action == 'LEFT':
            player_x -= PLAYER_SPEED
            player_facing_right = False
        elif current_action == 'RIGHT':
            player_x += PLAYER_SPEED
            player_facing_right = True

        # End game when all actions are completed
        if frame_count >= len(actions):
            running = False
            
        # Ensure character stays within map boundaries
        min_x = -WINDOW_WIDTH * 10
        max_x = len(buildings) * 150 - WINDOW_WIDTH // 2
        player_x = max(min_x, min(player_x, max_x))
        
        # Update camera offset
        camera_offset = player_x - WINDOW_WIDTH // 2
        
        # Check if character is in the middle of empty plot
        for building in buildings:
            if building.is_empty:
                building_center = building.x + building.width / 2
                if abs(player_x - building_center) < 20:
                    building.is_empty = False
                    building.growing = True
                    building.current_height = 0
                    building_count += 1
        
        # Update building heights
        for building in buildings:
            building.update()
        
        # Draw scene
        screen.fill(WHITE)
        
        # Draw buildings
        for building in buildings:
            screen_x = building.x - camera_offset
            if -building.width <= screen_x <= WINDOW_WIDTH:
                if not building.is_empty:
                    pygame.draw.rect(screen, building.color,
                                (screen_x, WINDOW_HEIGHT - building.current_height,
                                    building.width, building.current_height))
        
        # Draw counter
        font = pygame.font.Font(None, 180)
        count_text = font.render(f"{building_count}", True, (0, 0, 0))
        screen.blit(count_text, (WINDOW_WIDTH // 2 - 50, 20))
        
        # Draw character
        player_screen_x = WINDOW_WIDTH // 2
        pygame.draw.rect(screen, (0, 0, 0),
                        (player_screen_x, WINDOW_HEIGHT - PLAYER_HEIGHT - 10,
                        PLAYER_WIDTH, PLAYER_HEIGHT))
        
        # Convert screen content to image
        pygame_image = pygame.surfarray.array3d(screen)
        pygame_image = pygame_image.transpose(1, 0, 2)
        cv2_image = cv2.cvtColor(pygame_image, cv2.COLOR_RGB2BGR)
        
        # Resize image
        OUTPUT_WIDTH = 96
        OUTPUT_HEIGHT = 96
        cv2_image = cv2.resize(cv2_image, (OUTPUT_WIDTH, OUTPUT_HEIGHT), 
                            interpolation=cv2.INTER_AREA)
        
        # Save frame
        frame_count += 1
        next_action = actions[min(frame_count, len(actions) - 1)].lower()
        frame_name = f"{frame_count}_{next_action}_{building_count}.jpg"
        
        os.makedirs(f'data/0222_final_overall_dynamic_inference_dataset/output_{index}', exist_ok=True)
        cv2.imwrite(f'data/0222_final_overall_dynamic_inference_dataset/output_{index}/{frame_name}', cv2_image)
        
        pygame.time.wait(3)

indexes = list(range(100000))

if __name__ == '__main__':
    with Pool(processes=96) as p:
        with tqdm(total=len(indexes), desc='total') as pbar:
            for i, _ in enumerate(p.imap_unordered(make_sequence, indexes)):
                pbar.update()