# Core imports
import math
import numpy as np
import torch
import torch.nn as nn

# External imports
from timm.models.vision_transformer import PatchEmbed, Attention, Mlp

def modulate(x, shift, scale):
    return x * (1 + scale.unsqueeze(1)) + shift.unsqueeze(1)

#################################################################################
#                          Embedding Layers                                      #
#################################################################################

class TimestepEmbedder(nn.Module):
    """Embeds scalar timesteps into vector representations."""
    def __init__(self, hidden_size, frequency_embedding_size=256):
        super().__init__()
        self.mlp = nn.Sequential(
            nn.Linear(frequency_embedding_size, hidden_size, bias=True),
            nn.SiLU(),
            nn.Linear(hidden_size, hidden_size, bias=True),
        )
        self.frequency_embedding_size = frequency_embedding_size

    @staticmethod
    def timestep_embedding(t, dim, max_period=10000):
        """Creates sinusoidal timestep embeddings."""
        half = dim // 2
        freqs = torch.exp(
            -math.log(max_period) * torch.arange(start=0, end=half, dtype=torch.float32) / half
        ).to(device=t.device)
        args = t[:, None].float() * freqs[None]
        embedding = torch.cat([torch.cos(args), torch.sin(args)], dim=-1)
        if dim % 2:
            embedding = torch.cat([embedding, torch.zeros_like(embedding[:, :1])], dim=-1)
        return embedding

    def forward(self, t):
        t_freq = self.timestep_embedding(t, self.frequency_embedding_size)
        return self.mlp(t_freq)

class LabelEmbedder(nn.Module):
    """Embeds class labels into vector representations with optional dropout for classifier-free guidance."""
    def __init__(self, num_classes, hidden_size, dropout_prob):
        super().__init__()
        use_cfg_embedding = dropout_prob > 0
        self.embedding_table = nn.Embedding(num_classes + use_cfg_embedding, hidden_size)
        self.num_classes = num_classes
        self.dropout_prob = dropout_prob

    def token_drop(self, labels, force_drop_ids=None):
        """Implements label dropout for classifier-free guidance."""
        if force_drop_ids is None:
            drop_ids = torch.rand(labels.shape[0], device=labels.device) < self.dropout_prob
        else:
            drop_ids = force_drop_ids == 1
        labels = torch.where(drop_ids, self.num_classes, labels)
        return labels

    def forward(self, labels, train, force_drop_ids=None):
        use_dropout = self.dropout_prob > 0
        if (train and use_dropout) or (force_drop_ids is not None):
            labels = self.token_drop(labels, force_drop_ids)
        return self.embedding_table(labels)

class PositionalEncoding(nn.Module):
    """Implements positional encoding using sine and cosine functions."""
    def __init__(self, d_model, max_len=3000):
        super().__init__()
        position = torch.arange(max_len).unsqueeze(1)
        div_term = torch.exp(torch.arange(0, d_model, 2) * (-math.log(10000.0) / d_model))
        pe = torch.zeros(max_len, 1, d_model)
        pe[:, 0, 0::2] = torch.sin(position * div_term)
        pe[:, 0, 1::2] = torch.cos(position * div_term)
        self.register_buffer('pe', pe)

    def forward(self, x):
        return x + self.pe[:x.size(0)]

#################################################################################
#                          Core Model Components                                 #
#################################################################################

class DiTBlock(nn.Module):
    """Transformer block with adaptive layer norm zero (adaLN-Zero) conditioning."""
    def __init__(self, hidden_size, num_heads, mlp_ratio=4.0, **block_kwargs):
        super().__init__()
        self.norm1 = nn.LayerNorm(hidden_size, elementwise_affine=False, eps=1e-6)
        self.attn = Attention(hidden_size, num_heads=num_heads, qkv_bias=True, **block_kwargs)
        self.norm2 = nn.LayerNorm(hidden_size, elementwise_affine=False, eps=1e-6)
        mlp_hidden_dim = int(hidden_size * mlp_ratio)
        approx_gelu = lambda: nn.GELU(approximate="tanh")
        self.mlp = Mlp(in_features=hidden_size, hidden_features=mlp_hidden_dim, act_layer=approx_gelu, drop=0)
        self.adaLN_modulation = nn.Sequential(
            nn.SiLU(),
            nn.Linear(hidden_size, 6 * hidden_size, bias=True)
        )

    def forward(self, x, c):
        shift_msa, scale_msa, gate_msa, shift_mlp, scale_mlp, gate_mlp = self.adaLN_modulation(c).chunk(6, dim=1)
        x = x + gate_msa.unsqueeze(1) * self.attn(modulate(self.norm1(x), shift_msa, scale_msa))
        x = x + gate_mlp.unsqueeze(1) * self.mlp(modulate(self.norm2(x), shift_mlp, scale_mlp))
        return x

class FinalLayer(nn.Module):
    """Final layer of the model."""
    def __init__(self, hidden_size, patch_size, out_channels):
        super().__init__()
        self.norm_final = nn.LayerNorm(hidden_size, elementwise_affine=False, eps=1e-6)
        self.linear = nn.Linear(hidden_size, patch_size * patch_size * out_channels, bias=True)
        self.adaLN_modulation = nn.Sequential(
            nn.SiLU(),
            nn.Linear(hidden_size, 2 * hidden_size, bias=True)
        )

    def forward(self, x, c):
        shift, scale = self.adaLN_modulation(c).chunk(2, dim=1)
        x = modulate(self.norm_final(x), shift, scale)
        return self.linear(x)

class DiT(nn.Module):
    """
    Diffusion model with a Transformer backbone.
    """
    def __init__(
        self,
        input_size=32,
        patch_size=2,
        in_channels=4,
        hidden_size=1152,
        depth=28,
        num_heads=16,
        mlp_ratio=4.0,
        class_dropout_prob=0.1,
        num_classes=1000,
        learn_sigma=True,
    ):
        super().__init__()
        self.learn_sigma = learn_sigma
        self.in_channels = in_channels
        self.out_channels = in_channels * 2 if learn_sigma else in_channels
        self.patch_size = patch_size
        self.num_heads = num_heads
        self.hidden_size = hidden_size
        self.x_embedder = PatchEmbed(input_size, patch_size, in_channels, hidden_size, bias=True)
        num_patches = self.x_embedder.num_patches
        
        self.blocks = nn.ModuleList([
            DiTBlock(hidden_size, num_heads, mlp_ratio=mlp_ratio) for _ in range(depth)
        ])
        self.final_layer = FinalLayer(hidden_size, patch_size, self.out_channels)
        self.initialize_weights()
        self.pos_encoder = PositionalEncoding(384)

    def initialize_weights(self):
        """Initialize model weights."""
        def _basic_init(module):
            if isinstance(module, nn.Linear):
                torch.nn.init.xavier_uniform_(module.weight)
                if module.bias is not None:
                    nn.init.constant_(module.bias, 0)
        self.apply(_basic_init)

        # Initialize patch_embed like nn.Linear
        w = self.x_embedder.proj.weight.data
        nn.init.xavier_uniform_(w.view([w.shape[0], -1]))
        nn.init.constant_(self.x_embedder.proj.bias, 0)

        # Initialize modulation layers
        for block in self.blocks:
            nn.init.constant_(block.adaLN_modulation[-1].weight, 0)
            nn.init.constant_(block.adaLN_modulation[-1].bias, 0)

        # Initialize final layers
        nn.init.constant_(self.final_layer.adaLN_modulation[-1].weight, 0)
        nn.init.constant_(self.final_layer.adaLN_modulation[-1].bias, 0)
        nn.init.constant_(self.final_layer.linear.weight, 0)
        nn.init.constant_(self.final_layer.linear.bias, 0)

    def unpatchify(self, x):
        """
        Convert patched representation back to image.
        Args:
            x: (N, T, patch_size**2 * C) tensor of patch embeddings
        Returns:
            imgs: (N, C, H, W) tensor of images
        """
        c = self.out_channels
        p = self.x_embedder.patch_size[0]
        h = w = int(x.shape[1] ** 0.5)
        assert h * w == x.shape[1]

        x = x.reshape(shape=(x.shape[0], h, w, p, p, c))
        x = torch.einsum('nhwpqc->nchpwq', x)
        imgs = x.reshape(shape=(x.shape[0], c, h * p, h * p))
        return imgs

    def forward(self, x, c, digit_embeddings, maps):
        """
        Forward pass of DiT.
        Args:
            x: Input tensor of shape (N, C, H, W)
            c: Conditioning tensor
            digit_embeddings: Embedding tensor for digits
            maps: Additional feature maps
        """
        x = self.x_embedder(x)
        x = torch.cat([x, digit_embeddings, maps], dim=1)

        x = x.transpose(0, 1)
        x = self.pos_encoder(x)
        x = x.transpose(0, 1)

        for block in self.blocks:
            x = block(x, c)

        x = x[:, :-3-48, :]  # Remove auxiliary embeddings
        x = self.final_layer(x, c)
        x = self.unpatchify(x)
        return x

    def forward_with_cfg(self, x, t, y, cfg_scale):
        """
        Forward pass with classifier-free guidance.
        """
        half = x[: len(x) // 2]
        combined = torch.cat([half, half], dim=0)
        model_out = self.forward(combined, t, y)
        eps, rest = model_out[:, :3], model_out[:, 3:]
        cond_eps, uncond_eps = torch.split(eps, len(eps) // 2, dim=0)
        half_eps = uncond_eps + cfg_scale * (cond_eps - uncond_eps)
        eps = torch.cat([half_eps, half_eps], dim=0)
        return torch.cat([eps, rest], dim=1)

#################################################################################
#                   Positional Embedding Helper Functions                        #
#################################################################################

def get_2d_sincos_pos_embed(embed_dim, grid_size, cls_token=False, extra_tokens=0):
    """
    Generate 2D sinusoidal positional embeddings.
    Args:
        embed_dim: Output dimension for each position
        grid_size: Size of the grid (assumed to be square)
        cls_token: Include classification token
        extra_tokens: Number of extra tokens to append
    """
    grid_h = np.arange(grid_size, dtype=np.float32)
    grid_w = np.arange(grid_size, dtype=np.float32)
    grid = np.meshgrid(grid_w, grid_h)
    grid = np.stack(grid, axis=0)

    grid = grid.reshape([2, 1, grid_size, grid_size])
    pos_embed = get_2d_sincos_pos_embed_from_grid(embed_dim, grid)
    if cls_token and extra_tokens > 0:
        pos_embed = np.concatenate([np.zeros([extra_tokens, embed_dim]), pos_embed], axis=0)
    return pos_embed

def get_2d_sincos_pos_embed_from_grid(embed_dim, grid):
    """Generate 2D positional embeddings from a grid."""
    assert embed_dim % 2 == 0
    emb_h = get_1d_sincos_pos_embed_from_grid(embed_dim // 2, grid[0])
    emb_w = get_1d_sincos_pos_embed_from_grid(embed_dim // 2, grid[1])
    emb = np.concatenate([emb_h, emb_w], axis=1)
    return emb

def get_1d_sincos_pos_embed_from_grid(embed_dim, pos):
    """Generate 1D positional embeddings from positions."""
    assert embed_dim % 2 == 0
    omega = np.arange(embed_dim // 2, dtype=np.float64)
    omega /= embed_dim / 2.
    omega = 1. / 10000**omega

    pos = pos.reshape(-1)
    out = np.einsum('m,d->md', pos, omega)

    emb_sin = np.sin(out)
    emb_cos = np.cos(out)
    emb = np.concatenate([emb_sin, emb_cos], axis=1)
    return emb

#################################################################################
#                                Model Variants                                  #
#################################################################################

def DiT_XL_2(**kwargs):
    return DiT(depth=28, hidden_size=1152, patch_size=2, num_heads=16, **kwargs)

def DiT_XL_4(**kwargs):
    return DiT(depth=28, hidden_size=1152, patch_size=4, num_heads=16, **kwargs)

def DiT_XL_8(**kwargs):
    return DiT(depth=28, hidden_size=1152, patch_size=8, num_heads=16, **kwargs)

def DiT_L_2(**kwargs):
    return DiT(depth=24, hidden_size=1024, patch_size=2, num_heads=16, **kwargs)

def DiT_L_4(**kwargs):
    return DiT(depth=24, hidden_size=1024, patch_size=4, num_heads=16, **kwargs)

def DiT_L_8(**kwargs):
    return DiT(depth=24, hidden_size=1024, patch_size=8, num_heads=16, **kwargs)

def DiT_B_2(**kwargs):
    return DiT(depth=12, hidden_size=768, patch_size=2, num_heads=12, **kwargs)

def DiT_B_4(**kwargs):
    return DiT(depth=12, hidden_size=768, patch_size=4, num_heads=12, **kwargs)

def DiT_B_8(**kwargs):
    return DiT(depth=12, hidden_size=768, patch_size=8, num_heads=12, **kwargs)

def DiT_S_2(**kwargs):
    return DiT(depth=12, hidden_size=384, patch_size=2, num_heads=6, **kwargs)

def DiT_S_4(**kwargs):
    return DiT(depth=12, hidden_size=384, patch_size=4, num_heads=6, **kwargs)

def DiT_S_8(**kwargs):
    return DiT(depth=12, hidden_size=384, patch_size=8, num_heads=6, **kwargs)

def I64_S_8(**kwargs):
    return DiT(input_size=64,depth=12, hidden_size=384, patch_size=8, num_heads=6,learn_sigma=False, **kwargs)

def I64_S_4(**kwargs):
    return DiT(input_size=64,depth=12, hidden_size=384, patch_size=4, num_heads=6,learn_sigma=False, **kwargs)

def I64_S_2(**kwargs):
    return DiT(input_size=64,depth=12, hidden_size=384, patch_size=2, num_heads=6,learn_sigma=False, **kwargs)

def I64_B_8(**kwargs):
    return DiT(input_size=64,depth=12, hidden_size=768, patch_size=8, num_heads=12,learn_sigma=False, **kwargs)

def I64_B_4(**kwargs):
    return DiT(input_size=64,depth=12, hidden_size=768, patch_size=4, num_heads=12,learn_sigma=False, **kwargs)

def I64_B_2(**kwargs):
    return DiT(input_size=64,depth=12, hidden_size=768, patch_size=2, num_heads=12,learn_sigma=False, **kwargs)

def I64_L_8(**kwargs):
    return DiT(input_size=64,depth=24, hidden_size=1024, patch_size=8, num_heads=16,learn_sigma=False, **kwargs)

def I64_L_4(**kwargs):
    return DiT(input_size=64,depth=24, hidden_size=1024, patch_size=4, num_heads=16,learn_sigma=False, **kwargs)

def I64_L_2(**kwargs):
    return DiT(input_size=64,depth=24, hidden_size=1024, patch_size=2, num_heads=16,learn_sigma=False, **kwargs)

def I4_S_1(**kwargs):
    return DiT(input_size=4,depth=12, hidden_size=384, patch_size=1, num_heads=6,learn_sigma=False, **kwargs)

def I16_S_2(**kwargs):
    return DiT(input_size=16,depth=12, hidden_size=384, patch_size=2, num_heads=6,learn_sigma=False, **kwargs)

def I16_S_1(**kwargs):
    return DiT(input_size=16,depth=12, hidden_size=384, patch_size=1, num_heads=6,learn_sigma=False, **kwargs)
def I128_S_8(**kwargs):
    return DiT(input_size=128,depth=12, hidden_size=384, patch_size=8, num_heads=6,learn_sigma=False, **kwargs)

def I32_S_2(**kwargs):
    return DiT(input_size=32,depth=12, hidden_size=384, patch_size=2, num_heads=6,learn_sigma=False, **kwargs)

def I24_S_2(**kwargs):
    return DiT(input_size=24,depth=12, hidden_size=384, patch_size=2, num_heads=6,learn_sigma=False, **kwargs)

def I24_S_2_layer2(**kwargs):
    return DiT(input_size=24,depth=2, hidden_size=384, patch_size=2, num_heads=6,learn_sigma=False, **kwargs)

def I24_S_2_layer4(**kwargs):
    return DiT(input_size=24,depth=4, hidden_size=384, patch_size=2, num_heads=6,learn_sigma=False, **kwargs)

def I24_S_2_layer8(**kwargs):
    return DiT(input_size=24,depth=8, hidden_size=384, patch_size=2, num_heads=6,learn_sigma=False, **kwargs)



def I32_B_2(**kwargs):
    return DiT(input_size=32,depth=12, hidden_size=768, patch_size=2, num_heads=12,learn_sigma=False, **kwargs)

def I16_B_2(**kwargs):
    return DiT(input_size=16,depth=12, hidden_size=768, patch_size=2, num_heads=12,learn_sigma=False, **kwargs)

DiT_models = {
    'I64_S_2':  I64_S_2,   'I64_S_4':  I64_S_4,   'I64_S_8':  I64_S_8,
    'I64_B_2': I64_B_2, 'I64_B_4': I64_B_4, 'I64_B_8': I64_B_8,
    'I64_L_2': I64_L_2, 'I64_L_4': I64_L_4, 'I64_L_8': I64_L_8,
    'I4_S_1': I4_S_1,    'I128_S_8': I128_S_8, 'I16_S_1' : I16_S_1,
    'I16_S_2' : I16_S_2, 'I32_S_2': I32_S_2, 'I32_B_2': I32_B_2,
    'I24_S_2': I24_S_2, 'I24_S_2_layer2': I24_S_2_layer2, 
    'I24_S_2_layer4': I24_S_2_layer4, 'I24_S_2_layer8': I24_S_2_layer8,
}
