#!/usr/bin/env python3

import argparse
import os.path as path

from sarnn.utils import prune
from snn.utils import cleanup_tmp, load_model, register_handlers


def main(args):
    register_handlers()

    if not path.exists(args.model_filename):
        print("Could not find {}.".format(args.model_filename))
        print("Unable to proceed.")
        return

    out_filename = path.splitext(args.model_filename)[0] + "_" + args.id + ".h5"
    if path.exists(out_filename):
        print("File {} already exists.".format(out_filename))
        print("Delete it to process again.")
        return

    model = load_model(args.model_filename)
    prune(model, args.epsilon)
    model.save(out_filename)


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Zeros out kernel weights which are below epsilon.",
        formatter_class=argparse.ArgumentDefaultsHelpFormatter,
        add_help=False)

    parser.add_argument(
        "model_filename",
        help="The filename of the model to process. May refer to "
             "either an ANN or an SNN.")
    parser.add_argument(
        "-e", "--epsilon", default=1e-4, type=float,
        help="Kernel weights whose absolute values are less than "
             "epsilon are zeroed out.")
    parser.add_argument(
        "-I", "--id", default="z",
        help="A unique identifier to append to the name of the new "
             "model. This must be non-empty because the new model is "
             "written to the same directory as the old one.")
    parser.add_argument(
        "-h", "--help", action="help",
        help="Display this help message and exit.")

    try:
        main(parser.parse_args())
    finally:
        cleanup_tmp()
