#!/usr/bin/env python3

# %%

import os
import os.path as path

import matplotlib.pyplot as plt
import numpy as np

# %%

# There are 2 input neurons, each of which fires with probability 0.5 on
# each time step. The synapse weights are +-1. If the positive neuron
# fired on its own, it would be enough to trigger a downstream spike. If
# both neurons fire in sync, they cancel each other out. What happens on
# average when the inputs are random?

n_sim = 100000
t_max = 2000
v_t = np.zeros(shape=n_sim)  # Membrane potential "right now"
s = np.empty(shape=(n_sim, t_max))  # Spike history
v = np.empty(shape=(n_sim, t_max))  # Membrane potential history

rng = np.random.RandomState(seed=0)

for t in range(t_max):
    # Simulate non-leaky integrate and fire dynamics
    v_t += (rng.uniform(low=0, high=1, size=n_sim) > 0.5)  # +1 synapse
    v_t -= (rng.uniform(low=0, high=1, size=n_sim) > 0.5)  # -1 synapse
    s_t = (v_t >= 1)
    v_t -= s_t

    # Store results in history arrays
    s[:, t] = s_t
    v[:, t] = v_t

# %%

fig, ax = plt.subplots(nrows=2, ncols=1)
fig.set_tight_layout(True)

ax[0].plot(np.mean(s, axis=0))
ax[0].set(ylim=(0, None),
          xlabel="Time",
          ylabel="Spike Probability")

ax[1].plot(np.mean(v, axis=0))
ax[1].set(xlabel="Time",
          ylabel="Mean Membrane Potential")

base_dir = path.join("figures", "other")
os.makedirs(base_dir, exist_ok=True)
fig.savefig(path.join(base_dir, "poisson_convergence.pdf"))
fig.show()
