from __future__ import print_function

import traceback
import argparse
from argparse import RawTextHelpFormatter
import os

import sys
import time
import json
import random

SRUNNER_PATH = r'./scenario_runner'
sys.path.append(SRUNNER_PATH)
from srunner.tools.scenario_parser import ScenarioConfigurationParser
from scenarios import CostumeScenarioRunner


VERSION = '0.9.13'






def main():
    """
    main function
    """
    description = ("CARLA Scenario Runner: Setup, Run and Evaluate scenarios using CARLA\n"
                   "Current version: " + VERSION)

    # pylint: disable=line-too-long
    parser = argparse.ArgumentParser(description=description,
                                     formatter_class=RawTextHelpFormatter)
    parser.add_argument('-v', '--version', action='version', version='%(prog)s ' + VERSION)
    parser.add_argument('--host', default='127.0.0.1',
                        help='IP of the host server (default: localhost)')
    parser.add_argument('--port', default='2000',
                        help='TCP port to listen to (default: 2000)')
    parser.add_argument('--timeout', default="10.0",
                        help='Set the CARLA client timeout value in seconds')
    parser.add_argument('--trafficManagerPort', default='8000',
                        help='Port to use for the TrafficManager (default: 8000)')
    parser.add_argument('--trafficManagerSeed', default='0',
                        help='Seed used by the TrafficManager (default: 0)')
    parser.add_argument('--sync', action='store_true',
                        help='Forces the simulation to run synchronously')
    parser.add_argument('--list', action="store_true", 
                        help='List all supported scenarios and exit')

    parser.add_argument('--scenario', 
                        help='Name of the scenario to be executed. Use the preposition \'group:\' to run all scenarios of one class, e.g. ControlLoss or FollowLeadingVehicle')
    parser.add_argument('--openscenario', 
                        help='Provide an OpenSCENARIO definition')
    parser.add_argument('--openscenarioparams', 
                        help='Overwrited for OpenSCENARIO ParameterDeclaration')
    parser.add_argument('--route', nargs='+', type=str,
                        help='Run a route as a scenario (input: (route_file,scenario_file,[route id]))')

    parser.add_argument('--agent', 
                        help="Agent used to execute the scenario. Currently only compatible with route-based scenarios.")
    parser.add_argument('--agentConfig', type=str, default="",
                        help="Path to Agent's configuration file", )

    parser.add_argument('--output', action="store_true", 
                        help='Provide results on stdout')
    parser.add_argument('--file', action="store_true", 
                        help='Write results into a txt file')
    parser.add_argument('--junit', action="store_true", 
                        help='Write results into a junit file')
    parser.add_argument('--json', action="store_true", 
                        help='Write results into a JSON file')
    parser.add_argument('--outputDir', default='', 
                        help='Directory for output files (default: this directory)')

    parser.add_argument('--configFile', default='', 
                        help='Provide an additional scenario configuration file (*.xml)')
    parser.add_argument('--additionalScenario', default='', 
                        help='Provide additional scenario implementations (*.py)')

    parser.add_argument('--debug', action="store_true", 
                        help='Run with debug output')
    parser.add_argument('--reloadWorld', action="store_true",
                        help='Reload the CARLA world before starting a scenario (default=True)')
    parser.add_argument('--record', type=str, default='',
                        help='Path were the files will be saved, relative to SCENARIO_RUNNER_ROOT.\nActivates the CARLA recording feature and saves to file all the criteria information.')
    parser.add_argument('--randomize', action="store_true", 
                        help='Scenario parameters are randomized')
    parser.add_argument('--repetitions', default=1, type=int, 
                        help='Number of scenario executions')
    parser.add_argument('--waitForEgo', action="store_true", 
                        help='Connect the scenario to an existing ego vehicle')


    parser.add_argument('--save_dir', default='./out', type=str, 
                        help='Directory to save the collected data')
    parser.add_argument('--camera', action='store_true',
                        help='Enable camera sensors')
    parser.add_argument('--response-distance', default=100.0, type=float, 
                        help='Distance to track arond the ego vehicle')
    parser.add_argument('--extract-per-frames', default=1, type=int, 
                        help='Number of frames to extract one data point')
    parser.add_argument('--warmup-frames', default=30, type=int, 
                        help='Distance to track arond the ego vehicle')
    parser.add_argument('--max-episode-len', default=1000, type=int)
    parser.add_argument('--extra-walkers', default=0, type=int)   # TODO
    parser.add_argument('--extra-vehicles', default=0, type=int)   # TODO
    parser.add_argument('--video-data', action='store_true') # TODO
    parser.add_argument('--collsion-detector-ratio', '-cdr', default=0.5, type=float, 
                        help="Probability of disabling the collision detector. This is for simulating human drivers' misattention")

    

    arguments = parser.parse_args()
    # pylint: enable=line-too-long

    assert arguments.extract_per_frames >= 1

    if arguments.list:
        print("Currently the following scenarios are supported:")
        print(*ScenarioConfigurationParser.get_list_of_scenarios(arguments.configFile), sep='\n')
        return 1

    if not arguments.scenario and not arguments.openscenario and not arguments.route:
        print("Please specify either a scenario or use the route mode\n\n")
        parser.print_help(sys.stdout)
        return 1

    if arguments.route and (arguments.openscenario or arguments.scenario):
        print("The route mode cannot be used together with a scenario (incl. OpenSCENARIO)'\n\n")
        parser.print_help(sys.stdout)
        return 1

    if arguments.agent and (arguments.openscenario or arguments.scenario):
        print("Agents are currently only compatible with route scenarios'\n\n")
        parser.print_help(sys.stdout)
        return 1

    if arguments.openscenarioparams and not arguments.openscenario:
        print("WARN: Ignoring --openscenarioparams when --openscenario is not specified")

    if arguments.route:
        arguments.reloadWorld = True

    if arguments.agent:
        arguments.sync = True

    scenario_runner = None
    result = True

    # random.seed(int(arguments.trafficManagerSeed))

    start_time = time.time()

    try:
        scenario_runner = CostumeScenarioRunner(arguments)
        result = scenario_runner.run()
    except Exception:   # pylint: disable=broad-except
        traceback.print_exc()

    finally:
        if scenario_runner is not None:
            scenario_runner.destroy()
            del scenario_runner

    print(time.time() - start_time)

    return not result


if __name__ == "__main__":
    sys.exit(main())



# export SCENARIO_RUNNER_ROOT=/home/bar/xb/research/av/git_repo/av/roadscene2vec/data/gen2/scenario_runner


# python main.py --scenario group:FollowLeadingVehicle --reloadWorld --sync --warmup-frames 50 --extract-per-frames 5 --repetition 5

# python main.py --scenario VehicleTurningRight_1 --reloadWorld --sync --warmup-frames 50 --extract-per-frames 5 --repetition 1 --max-episode-len 400 --extra-walkers 20
# export PYTHONPATH="$PYTHONPATH:/home/bar/xb/research/av/CARLA/PythonAPI/carla/dist/carla-0.9.13-py3.7-linux-x86_64.egg"