import torch
import torch.nn as nn
import torch.nn.functional as F


class DoubleConv(nn.Module):
    # This function preform double convolution with batch normalization
    # and ReLU activation function after each one of them.
    # DoubleConv = (convolution => [BN] => ReLU) * 2

    def __init__(self, in_channels, out_channels):
        super(DoubleConv, self).__init__()
        self.double_conv = nn.Sequential(
            nn.Conv2d(in_channels, out_channels, kernel_size=3, padding=1), # Convolution layer
            nn.BatchNorm2d(out_channels), # Batch normalization
            nn.ReLU(inplace=True), # ReLU activation function
            nn.Conv2d(out_channels, out_channels, kernel_size=3, padding=1), # Convolution layer
            nn.BatchNorm2d(out_channels), # Batch normalization
            nn.ReLU(inplace=True) # ReLU activation function
        )

    def forward(self, x):
        return self.double_conv(x)


class Down(nn.Module):
    # This function preform max pooling 2x2 and double convolution layers.
    # Down = downscaling with maxpool and then double conv

    def __init__(self, in_channels, out_channels):
        super(Down, self).__init__()
        self.maxpool_conv = nn.Sequential(
            nn.MaxPool2d(2), # Max pooling 2x2
            DoubleConv(in_channels, out_channels) # DoubleConv
        )

    def forward(self, x):
        return self.maxpool_conv(x)


class Up(nn.Module):
    # This function preform upscaling 2x2 and double convolution layers.
    # Up = upscaling and then double conv

    def __init__(self, in_channels, out_channels):
        super(Up, self).__init__()

        self.up = nn.ConvTranspose2d(in_channels, in_channels // 2, kernel_size=2, stride=2) # Upscaling 2x2
        self.conv = DoubleConv(in_channels, out_channels) # DoubleConv

    def forward(self, x1, x2):
        # In this function we can see the using with skip connections by the input x2
        x1 = self.up(x1)
        # Merge between the input of this layer - x1 and the input from the skip connection - x2
        diffY = x2.size()[2] - x1.size()[2]
        diffX = x2.size()[3] - x1.size()[3]

        x1 = F.pad(x1, [diffX // 2, diffX - diffX // 2,
                        diffY // 2, diffY - diffY // 2])

        # if you have padding issues, see
        # https://github.com/HaiyongJiang/U-Net-Pytorch-Unstructured-Buggy/commit/0e854509c2cea854e247a9c615f175f76fbb2e3a
        # https://github.com/xiaopeng-liao/Pytorch-UNet/commit/8ebac70e633bac59fc22bb5195e513d5832fb3bd

        x = torch.cat([x2, x1], dim=1)
        return self.conv(x)


class OutConv(nn.Module):
    # The function preform convolution with kernel_size=1
    def __init__(self, in_channels, out_channels):
        super(OutConv, self).__init__()
        self.conv = nn.Conv2d(in_channels, out_channels, kernel_size=1)

    def forward(self, x):
        return self.conv(x)


class UNet(nn.Module):
    # This function implement the network model (the model described in figure 5)

    def __init__(self, n_channels, n_classes):
        super(UNet, self).__init__()
        self.n_channels = n_channels  # Number of channels in the input images - for gray scale its equal to 1.
        self.n_classes = n_classes  # Number of channels in the output images - for gray scale its equal to 1.

        # The network layers (there is a option to use dropuots - we did not use it)
        self.inc = DoubleConv(n_channels, 64)
        self.down1 = Down(64, 128)
        self.down2 = Down(128, 256)
        # self.drop2 = nn.Dropout(0.5)
        self.down3 = Down(256, 512)
        # self.drop3 = nn.Dropout(0.5)
        self.down4 = Down(512, 1024)
        # self.drop4 = nn.Dropout(0.5)
        self.up1 = Up(1024, 512)
        # self.drop5 = nn.Dropout(0.5)
        self.up2 = Up(512, 256)
        # self.drop6 = nn.Dropout(0.5)
        self.up3 = Up(256, 128)
        self.up4 = Up(128, 64)
        self.outc = OutConv(64, n_classes)

    def forward(self, x):
        x1 = self.inc(x)
        x2 = self.down1(x1)
        x3 = self.down2(x2)
        # x34 = self.drop2(x3)
        x4 = self.down3(x3)
        # x45 = self.drop3(x4)
        x5 = self.down4(x4)
        # x56 = self.drop4(x5)

        x = self.up1(x5, x4)
        # x = self.drop5(x)
        x = self.up2(x, x3)
        # x = self.drop5(x)
        x = self.up3(x, x2)
        x = self.up4(x, x1)
        logits = self.outc(x)
        return logits
