"""Copyright 2021 ETH Zurich, Seonwook Park

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
"""
import gzip
import logging
import os
import pickle
import time
import zipfile
from tqdm import tqdm
import numpy as np
import torch

from config.config_default import DefaultConfig
from datasources import EVESequences_test, EVESequences_val
from models import STGazeNet, to_screen_coordinates, STGazeNetAblation


# Default singleton config object
config = DefaultConfig()

# Setup logger
logger = logging.getLogger(__name__)

# Set device
device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")

validation = False

def script_init_common():
    # Set inference-specific overrides
    config.override('fully_reproducible', True)
    config.override('refine_net_enabled', False)
    config.override('load_screen_content', False)
    config.override('do_full_test', True)
    config.override('load_full_frame_for_visualization', False)
    config.override('max_sequence_len', 30)
    config.override('start_time', 0)
    config.override('assumed_frame_rate', 10)
    config.override('start_frame', int(config.start_time * config.assumed_frame_rate))
    if os.path.basename(config.datasrc_eve) != "eve_dataset":
        config.override("datasrc_eve", os.path.join(os.path.dirname(config.datasrc_eve), "eve_dataset"))
        os.rmdir("segmentation_cache") 

def init_dataset(validation=False):
    # Initialize dataset and dataloader
    if validation:
        dataset = EVESequences_val(
            config.datasrc_eve,
            is_final_test=False,
        )
    else:
        dataset = EVESequences_test(
            config.datasrc_eve,
            is_final_test=True,
        )
        
    dataloader = torch.utils.data.DataLoader(
        dataset,
        batch_size=config.codalab_eval_batch_size,
        shuffle=False,
        drop_last=False,
        num_workers=config.codalab_eval_data_workers,
        pin_memory=True,
    )

    return dataset, dataloader

def feed_model(model, full_input_dict):
    batch_size = next(iter(full_input_dict.values())).shape[0]
    sequence_len = next(iter(full_input_dict.values())).shape[1]
    intermediate_dicts = []
    hidden_states = {"left": None, "right": None}
    for t in range(sequence_len):
        sub_input_dict = {k: v[:, t] for k, v in full_input_dict.items() if isinstance(v, torch.Tensor)}
        sub_output_dict = {}
        face_origin = torch.zeros_like(sub_input_dict["face_o"])
        face_rotation = torch.zeros_like(sub_input_dict["face_R"])
        for side in config.sides:
            hidden_state = hidden_states[side]
            flip = config.flip_right_eye and side == "right"
            gaze_prediction, hidden_state = model(sub_input_dict, side, hidden_state, flip)
            sub_output_dict[f"{side}_gaze"] = gaze_prediction
            hidden_states[side] = hidden_state
            origin = sub_input_dict[f"{side}_o"]
            rotation = sub_input_dict[f"{side}_R"]
            inv_camera_transformation = sub_input_dict["inv_camera_transformation"]
            pixels_per_millimeter = sub_input_dict["pixels_per_millimeter"]
            PoG_mm, PoG_px = to_screen_coordinates(origin, gaze_prediction, rotation, inv_camera_transformation, pixels_per_millimeter)
            sub_output_dict[f"{side}_PoG_mm"] = PoG_mm
            sub_output_dict[f"{side}_PoG_px"] = PoG_px
            face_origin += origin / 2
            face_rotation += rotation / 2
        sub_output_dict["PoG_px_final"] = (sub_output_dict["left_PoG_px"] + sub_output_dict["right_PoG_px"]) / 2
        gaze_prediction = (sub_output_dict["left_gaze"] + sub_output_dict["right_gaze"]) / 2
        _, PoG_px_initial = to_screen_coordinates(face_origin, gaze_prediction, face_rotation, inv_camera_transformation, pixels_per_millimeter)
        sub_output_dict["PoG_px_initial"] = PoG_px_initial
        intermediate_dicts.append(sub_output_dict)
    intermediate_dict = {k: torch.stack([d[k] for d in intermediate_dicts], dim=1) for k in intermediate_dicts[0].keys()}
    intermediate_dict['left_pupil_size'] = torch.zeros(batch_size, sequence_len, device=device)
    intermediate_dict['right_pupil_size'] = intermediate_dict['left_pupil_size']
    intermediate_dict['timestamps'] = full_input_dict['timestamps']

    return intermediate_dict

def iterator(model, dataloader):
    model.eval()
    with torch.no_grad():
        for current_step, input_data in enumerate(dataloader):
            loss_dict = None

            # Move tensors to device
            input_data_gpu = {}
            for k, v in input_data.items():
                if isinstance(v, torch.Tensor):
                    input_data_gpu[k] = v.detach().to(device, non_blocking=True)

            # Forward pass and yield
            outputs = feed_model(model, input_data_gpu)
            if validation:
                loss_dict = model.loss(input_data_gpu, outputs)

            # Convert data
            inputs_np = {
                k: v.numpy() if isinstance(v, torch.Tensor) else v
                for k, v in input_data.items()
            }
            outputs_np = {
                k: v.detach().cpu().numpy()
                for k, v in outputs.items()
                if isinstance(v, torch.Tensor)
            }
            yield current_step, inputs_np, outputs_np, loss_dict

def model_setup(model):
    # Load pre-trained model weights
    model.load_state_dict(
        torch.load(config.st_net_weights, map_location=device),
        strict=True
    )
    model.to(device)
    model.eval()
    return model

script_init_common()

dataset, dataloader = init_dataset(validation)

if config.st_net_ablation:
    model = STGazeNetAblation(
        eye_encoder=config.ablation_eye_encoder,
        face_encoder=config.ablation_face_encoder,
        eca=config.ablation_eca,
        sam=config.ablation_sam,
        gru=config.ablation_gru,
    )
else:
    model = STGazeNet()
model = model_setup(model)

max_dataset_len = len(dataset)
batch_size = config.codalab_eval_batch_size if not validation else config.full_test_batch_size
num_steps = int((max_dataset_len // batch_size) + (max_dataset_len % batch_size > 0))

processed_so_far = set()
outputs_to_write = {}
if validation:
    loss_dicts = []
progress_bar = tqdm(total=num_steps, desc="Processing", unit="batch")
for step, inputs, outputs, loss_dict in iterator(model, dataloader):

    if validation:
        loss_dicts.append(loss_dict)
    batch_size = next(iter(outputs.values())).shape[0]

    for i in range(batch_size):
        participant = inputs['participant'][i]
        subfolder = inputs['subfolder'][i]
        camera = inputs['camera'][i]
        # print('Processing %s/%s/%s' % (participant, subfolder, camera))

        # Ensure that the sub-dicts exist.
        if participant not in outputs_to_write:
            outputs_to_write[participant] = {}
        if subfolder not in outputs_to_write[participant]:
            outputs_to_write[participant][subfolder] = {}

        # Store back to output structure
        keys_to_store = [
            'timestamps',
            'left_pupil_size',
            'right_pupil_size',
            'PoG_px_initial',
            'PoG_px_final',
        ]
        sub_dict = outputs_to_write[participant][subfolder]
        if camera in sub_dict:
            for key in keys_to_store:
                sub_dict[camera][key] = np.concatenate([sub_dict[camera][key],
                                                        outputs[key][i, :]], axis=0)
        else:
            sub_dict[camera] = {}
            for key in keys_to_store:
                sub_dict[camera][key] = outputs[key][i, :]

        sequence_key = (participant, subfolder, camera)
        if sequence_key not in processed_so_far:
            # print('Handling %s/%s/%s' % sequence_key)
            processed_so_far.add(sequence_key)
    if validation:
        progress_bar.set_postfix({"loss ang": loss_dict['loss_ang'].mean().item(), "loss pog": loss_dict['loss_pog_px'].mean().item()})
    progress_bar.update(1)

if validation:
    loss_dict = {k: torch.stack([d[k] for d in loss_dicts], dim=0) for k in loss_dicts[0].keys()}
    for k, v in loss_dict.items():
        print(f"{k}: {v.mean().item()}")

def check_and_convert(obj, prefix=""):
    if isinstance(obj, np.ndarray):
        return obj.tolist()
    elif isinstance(obj, dict):
        return {k: check_and_convert(v, prefix=f"{prefix}{k}->") for k, v in obj.items()}
    elif isinstance(obj, list):
        return [check_and_convert(v, prefix=f"{prefix}[{i}]->") for i, v in enumerate(obj)]
    elif isinstance(obj, tuple):
        return tuple(check_and_convert(v, prefix=f"{prefix}({i})->") for i, v in enumerate(obj))
    else:
        return obj
    
def to_numpy(obj):
    if isinstance(obj, list):
        try:
            return np.array(obj)
        except:
            return [to_numpy(v) for v in obj]
    elif isinstance(obj, dict):
        return {k: to_numpy(v) for k, v in obj.items()}
    elif isinstance(obj, tuple):
        return tuple(to_numpy(v) for v in obj)
    else:
        return obj

if not validation:
    outputs_to_write = check_and_convert(outputs_to_write)
    #  Write output file
    model_name, run_name, weights_name = config.st_net_weights.split("/")[-3:]
    weights_name = weights_name.split(".")[0]
    output_fname = f"codalab/{model_name}_{run_name}_{weights_name}.pkl.gz"
    final_output_path = os.path.join(os.getcwd(), output_fname)
    with gzip.open(final_output_path, 'wb') as f:
        pickle.dump(outputs_to_write, f, protocol=3)
    print('Output file %s created' % final_output_path)