import tqdm
import copy
import time
import wandb
import torch
import numpy as np
from torch.optim import *
import torch.nn.functional as F

# self-defined functions
from models import eval_help
from data_preprocessing import Dataset
from utils import FedAvg_agg, FedOpt_agg, grad_norm

def SSL(args: object, clients: list[object], server: object, num_sample_client: int) -> None:
    """
    Main loop for vanilla split learning, which is also called sequential split learning.

    Arguments:
        args (argparse.Namespace): parsed argument object.
        clients (list[Client]): list of clients.
        server (Server): server.
        num_sample_client (int): how many clients are involved in one round.
    """

    # in SSL, every client trains the latest model
    latest_client_model = copy.deepcopy(clients[0].model)
    
    # evaluate model performance before training
    eval_help(args, clients, server)
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        lag, grad_norms, grad_norms_per_para = 0, [], []
        for c in sample_clients:
            # client side
            c.model.load_state_dict(latest_client_model.state_dict())
            x, y = c.forward_one_batch()
            
            # server side
            grad, elapsed = server.grad_one_batch(x, y)
            server.apply_optim()
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            # client side
            c.backward_one_batch(grad)
            c.apply_optim()
            latest_client_model.load_state_dict(c.model.state_dict())
        
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
        
        # evaluation
        eval_help(args, sample_clients, server, wandb_log)
    
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def PSL(args: object, clients: list[object], server: object, num_sample_client: int) -> None:
    """
    Main loop for parallel split learning.

    Arguments:
        same as SSL.
    """

    # in PSL, server has multiple copies of server part model. We emulate this by having multiple servers.
    servers = [copy.deepcopy(server) for i in range(num_sample_client)]

    # evaluate model performance before training
    eval_help(args, clients, servers[0])
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        lag, grad_norms, grad_norms_per_para = 0, [], []
        for c, s in zip(sample_clients, servers):
            # client side
            x, y = c.forward_one_batch()
            
            # server side
            grad, elapsed = s.grad_one_batch(x, y)
            s.apply_optim()
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            # client side
            c.backward_one_batch(grad)
            c.apply_optim()
        
        # aggregate server models
        start = time.perf_counter()
        FedAvg_agg(servers, weights = [c.num_sample for c in sample_clients])
        end = time.perf_counter()
        lag += end - start
        
        # log lag and grad norm
        # log grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
                    
        # evaluation. For a fair comparison of model generalization, we use local model of a random client as client part.
        # note that all server models are identical
        eval_help(args, sample_clients, servers[0], wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def SFLV1(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    """
    Main loop for split federated learning (SplitFed) version 1.

    Arguments:
        same as SSL.
    """

    # in SFLV1, server has multiple copies of server part model. We emulate this by having multiple servers.
    servers = [copy.deepcopy(server) for i in range(num_sample_client)]

    # evaluate model performance before training
    eval_help(args, clients, servers[0])
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        lag, grad_norms, grad_norms_per_para = 0, [], []
        for c, s in zip(sample_clients, servers):
            # client side
            x, y = c.forward_one_batch()
            
            # server side
            grad, elapsed = s.grad_one_batch(x, y)
            s.apply_optim()
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            # client side
            c.backward_one_batch(grad)
            c.apply_optim()
            
        # aggregate server models
        start = time.perf_counter()
        FedAvg_agg(servers, weights = [c.num_sample for c in sample_clients])
        end = time.perf_counter()
        lag += end - start
        
        # aggregate client models
        if agg_all_clients: # aggregation influences all clients (also those clients who are not sampled in this round)
            FedAvg_agg(sample_clients, clients, weights = [c.num_sample for c in sample_clients])
        else: # aggregation influences only sampled clients
            FedAvg_agg(sample_clients, weights = [c.num_sample for c in sample_clients])
        
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
            
        # evaluation. Note that all client models are identical and all server models are identical.
        eval_help(args, sample_clients, servers[0], wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def SFLV2(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    """
    Main loop for split federated learning (SplitFed) version 2.

    Arguments:
        same as SSL.
    """

    # evaluate model performance before training
    eval_help(args, clients, server)
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        lag, grad_norms, grad_norms_per_para = 0, [], []
        for c in sample_clients:
            # client side
            x, y = c.forward_one_batch()
            
            # server side
            grad, elapsed = server.grad_one_batch(x, y)
            server.apply_optim()
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            # client side
            c.backward_one_batch(grad)
            c.apply_optim()
            
        # aggregate client models
        if agg_all_clients: # aggregation influences all clients (also those clients who are not sampled in this round)
            FedAvg_agg(sample_clients, clients, weights = [c.num_sample for c in sample_clients])
        else: # aggregation influences only sampled clients
            FedAvg_agg(sample_clients, weights = [c.num_sample for c in sample_clients])
            
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
            
        # evaluation. Note that all client models are identical and all server model are identical.
        eval_help(args, sample_clients, server, wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def SGLR(args: object, clients: list[object], server: object, num_sample_client: int, active_C: float = 0.5) -> None:
    """
    Main loop for server-Side local gradient averaging and learning rate acceleration (SGLR, which combines SLR and SGL). 

    Arguments:
        same as SSL.
    """

    # in SGLR, server has multiple copies of server part model. We emulate this by having multiple servers.
    servers = [copy.deepcopy(server) for i in range(num_sample_client)]

    # evaluate model performance before training
    eval_help(args, clients, servers[0])

    # active average clients in SGLR
    num_active_client = int(num_sample_client * active_C)
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        i, grads, weights = 0, [], []
        lag, grad_norms, grad_norms_per_para = 0, [], []
        for c, s in zip(sample_clients, servers):
            # client side
            x, y = c.forward_one_batch()
            
            # server side
            grad, elapsed = s.grad_one_batch(x, y)
            s.apply_optim()
            lag += elapsed

            if i < num_active_client: # active clients
                c.to('cpu')
                grads.append(grad)
                weights.append(c.num_sample)
            else: # inactive clients
                # log grad norm
                g_norm, g_norm_per_para = grad_norm(grad)
                grad_norms.append(g_norm)
                grad_norms_per_para.append(g_norm_per_para)
                all_grad_norms.append(g_norm.cpu())
            
                c.backward_one_batch(grad)
                c.apply_optim()
            i += 1

        # aggregate server models
        start = time.perf_counter()
        FedAvg_agg(servers, weights = [c.num_sample for c in sample_clients])

        # average gradients
        sum_grad, sum_w = 0, 0
        for grad, w in zip(grads, weights):
            # unroll batch
            for sample_grad in grad:
                sum_grad += sample_grad * w
                sum_w += w
        avg_grad = sum_grad / sum_w
        
        end = time.perf_counter()
        lag += end - start

        g_norm = avg_grad.norm()
        g_norm_per_para = g_norm / (avg_grad.numel() ** 0.5)
            
        # update active clients
        for c in sample_clients[:num_active_client]:
            # log grad norm
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            c.to(args.device)
            c.backward_one_batch(avg_grad)
            c.apply_optim()
            
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client    
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
                        
        # evaluation. Note all server models are identical.
        eval_help(args, sample_clients, servers[0], wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def FedAvg(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    """
    Main loop for vanilla federated learning (FedAvg).

    Arguments:
        same as SSL.
    """

    # we emulate FL (FedAvg) with client-server pairs. A pair has a full model.
    servers = [copy.deepcopy(server) for c in clients]
    
    # evaluate model performance before training
    eval_help(args, clients, servers[0])
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample pairs which are involved in this round
        sample_idx = np.random.choice(len(clients), num_sample_client, replace = False)
        sample_clients = [clients[i] for i in sample_idx]
        sample_servers = [servers[i] for i in sample_idx]
        
        lag, grad_norms, grad_norms_per_para = 0, [], []
        for c, s in zip(sample_clients, sample_servers):
            x, y = c.forward_one_batch()
            
            grad, elapsed = s.grad_one_batch(x, y)
            s.apply_optim()
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            c.backward_one_batch(grad)
            c.apply_optim()

        # aggregation
        if agg_all_clients: # aggregation influences all clients (also those clients who are not sampled in this round)
            FedAvg_agg(sample_clients, clients, weights = [c.num_sample for c in sample_clients])
            start = time.perf_counter()
            FedAvg_agg(sample_servers, servers, weights = [c.num_sample for c in sample_clients])
            end = time.perf_counter()
        else: # aggregation influences only sampled clients
            FedAvg_agg(sample_clients, weights = [c.num_sample for c in sample_clients])
            start = time.perf_counter()
            FedAvg_agg(sample_servers, weights = [c.num_sample for c in sample_clients])
            end = time.perf_counter()
            
        # log lag and grad norm
        wandb_log = {}
        lag += end - start
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
        
        # evaluation
        eval_help(args, sample_clients, sample_servers[0], wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def CyclePSL(args: object, clients: list[object], server: object, num_sample_client: int) -> None:
    """
    Main loop for our method + PSL.

    Arguments:
        same as SSL.
    """

    # evaluate model performance before training
    eval_help(args, clients, server)
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        # feature datasets
        feature_datasets, lag = [], 0
        for c in sample_clients:
            # client side
            x, y = c.forward_one_batch()
            x = x.to('cpu')
            y = y.to('cpu')
            c.to('cpu')
            
            # collect features, labels, and weights from clients, and form datasets
            start = time.perf_counter()
            d = Dataset(x, y, c.num_sample)
            feature_datasets.append(d)
            end = time.perf_counter()
            lag += end - start
        
        # form dataset and dataloader for 2nd level machine learning (expectation maximization)
        start = time.perf_counter()
        server_dataset = torch.utils.data.ConcatDataset(feature_datasets)
        server_loader  = torch.utils.data.DataLoader(server_dataset, batch_size = args.server_bs, shuffle = True)
        end = time.perf_counter()
        lag += end - start
        
        # 2nd level ML on server side
        server.to(args.device)
        server.freeze(False)
        server.model.train()
        start = time.perf_counter()
        for current_server_epoch in range(args.num_server_epoch):
            for batch_id, (x, y, w) in enumerate(server_loader):
                x = x.to(args.device)
                y = y.to(args.device)
                w = w.to(args.device)

                server.optim.zero_grad()
                preds = server.model(x)
                
                # weight loss
                if args.binary is None: # regression, for openeds2020
                    loss = 1 - F.cosine_similarity(preds, y)
                else: # classification, no matter True of False
                    loss = F.cross_entropy(preds, y, reduction = 'none')
                loss = sum(loss * w) / sum(w)
                
                loss.backward()
                server.optim.step()
            
        # freeze server model and re-use client features to optimize client models
        server.model.zero_grad()
        server.freeze(True)
        
        end = time.perf_counter()
        lag += end - start
        
        grad_norms, grad_norms_per_para = [], []
        for c, d in zip(sample_clients, feature_datasets):
            grad, elapsed = server.grad_one_batch(d.xs.to(args.device), d.ys.to(args.device))
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            c.to(args.device)
            c.backward_one_batch(grad)
            c.apply_optim()
            
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client    
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)

        # evaluation
        eval_help(args, sample_clients, server, wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def CycleSFL(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    """
    Main loop for our method + SFL / FSL.

    Arguments:
        same as SSL.
    """

    # evaluate model performance before training
    eval_help(args, clients, server)
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        # feature datasets
        feature_datasets, lag = [], 0
        for c in sample_clients:
            # client side
            x, y = c.forward_one_batch()
            x = x.to('cpu')
            y = y.to('cpu')
            c.to('cpu')
            
            # collect features, labels, and weights from clients, and form datasets
            start = time.perf_counter()
            d = Dataset(x, y, c.num_sample)
            feature_datasets.append(d)
            end = time.perf_counter()
            lag += end - start
        
        # form dataset and dataloader for 2nd level machine learning (expectation maximization)
        start = time.perf_counter()
        server_dataset = torch.utils.data.ConcatDataset(feature_datasets)
        server_loader  = torch.utils.data.DataLoader(server_dataset, batch_size = args.server_bs, shuffle = True)
        end = time.perf_counter()
        lag += end - start
        
        # higher level ML on server side
        server.to(args.device)
        server.freeze(False)
        server.model.train()
        start = time.perf_counter()
        for current_server_epoch in range(args.num_server_epoch):
            for batch_id, (x, y, w) in enumerate(server_loader):
                x = x.to(args.device)
                y = y.to(args.device)
                w = w.to(args.device)

                server.optim.zero_grad()
                preds = server.model(x)
                
                # weight loss
                if args.binary is None: # regression, for openeds2020
                    loss = 1 - F.cosine_similarity(preds, y)
                else: # classification, no matter True of False
                    loss = F.cross_entropy(preds, y, reduction = 'none')
                loss = sum(loss * w) / sum(w)
                
                loss.backward()
                server.optim.step()
                    
        # freeze server model and re-use client features to optimize client models
        server.model.zero_grad()
        server.freeze(True)
        
        end = time.perf_counter()
        lag += end - start
        
        grad_norms, grad_norms_per_para = [], []
        for c, d in zip(sample_clients, feature_datasets):
            grad, elapsed = server.grad_one_batch(d.xs.to(args.device), d.ys.to(args.device))
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            c.to(args.device)
            c.backward_one_batch(grad)
            c.apply_optim()
        
        # aggregate client models
        if agg_all_clients: # aggregation influences also not sampled clients
            FedAvg_agg(sample_clients, clients, weights = [c.num_sample for c in sample_clients])
        else: # aggregation influences only sampled clients
            FedAvg_agg(sample_clients, weights = [c.num_sample for c in sample_clients])
            
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
        
        # evaluation
        eval_help(args, sample_clients, server, wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def CycleSGLR(args: object, clients: list[object], server: object, num_sample_client: int, active_C: float = 0.5) -> None:
    """
    Main loop for our method + SGLR.

    Arguments:
        same as SSL.
    """

    # evaluate model performance before training
    eval_help(args, clients, server)

    # active average clients in SGLR
    num_active_client = int(num_sample_client * active_C)
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        # feature datasets
        feature_datasets, lag = [], 0
        for c in sample_clients:
            # client side
            x, y = c.forward_one_batch()
            x = x.to('cpu')
            y = y.to('cpu')
            c.to('cpu')
            
            # collect features, labels, and weights from clients, and form datasets
            start = time.perf_counter()
            d = Dataset(x, y, c.num_sample)
            feature_datasets.append(d)
            end = time.perf_counter()
            lag += end - start
        
        # form dataset and dataloader for 2nd level machine learning (expectation maximization)
        start = time.perf_counter()
        server_dataset = torch.utils.data.ConcatDataset(feature_datasets)
        server_loader  = torch.utils.data.DataLoader(server_dataset, batch_size = args.server_bs, shuffle = True)
        end = time.perf_counter()
        lag += end - start
        
        # 2nd level ML on server side
        server.to(args.device)
        server.freeze(False)
        server.model.train()
        start = time.perf_counter()
        for current_server_epoch in range(args.num_server_epoch):
            for batch_id, (x, y, w) in enumerate(server_loader):
                x = x.to(args.device)
                y = y.to(args.device)
                w = w.to(args.device)

                server.optim.zero_grad()
                preds = server.model(x)
                
                # weight loss
                if args.binary is None: # regression, for openeds2020
                    loss = 1 - F.cosine_similarity(preds, y)
                else: # classification, no matter True of False
                    loss = F.cross_entropy(preds, y, reduction = 'none')
                loss = sum(loss * w) / sum(w)
                
                loss.backward()
                server.optim.step()
                
        # freeze server model and re-use client features to optimize client models
        server.model.zero_grad()
        server.freeze(True)
        
        end = time.perf_counter()
        lag += end - start
        
        i, grads, weights = 0, [], []
        grad_norms, grad_norms_per_para = [], []
        for c, d in zip(sample_clients, feature_datasets):
            grad, elapsed = server.grad_one_batch(d.xs.to(args.device), d.ys.to(args.device))
            lag += elapsed
            
            if i < num_active_client: # active clients
                grads.append(grad.to('cpu'))
                weights.append(c.num_sample)
            else: # inactive clients
                # log grad norm
                g_norm, g_norm_per_para = grad_norm(grad)
                grad_norms.append(g_norm)
                grad_norms_per_para.append(g_norm_per_para)
                all_grad_norms.append(g_norm.cpu())
                
                c.to(args.device)
                c.backward_one_batch(grad)
                c.apply_optim()
            i += 1

        # average gradients
        start = time.perf_counter()
        sum_grad, sum_w = 0, 0
        for grad, w in zip(grads, weights):
            # unroll batch
            for sample_grad in grad:
                sum_grad += sample_grad * w
                sum_w += w
        avg_grad = sum_grad / sum_w
        avg_grad = avg_grad.to(args.device)
        end = time.perf_counter()
        lag += end - start
        
        g_norm = avg_grad.norm()
        g_norm_per_para = g_norm / (avg_grad.numel() ** 0.5)

        # update active clients
        for c in sample_clients[:num_active_client]:
            # log grad norm
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            c.to(args.device)
            c.backward_one_batch(avg_grad)
            c.apply_optim()
            
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)

        # evaluation
        eval_help(args, sample_clients, server, wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

def CycleSSL(args: object, clients: list[object], server: object, num_sample_client: int) -> None:
    """
    Main loop for our method + SSL.

    Arguments:
        same as SSL.
    """

    # in SSL, every client trains the latest model
    latest_client_model = copy.deepcopy(clients[0].model)

    # evaluate model performance before training
    eval_help(args, clients, server)
    
    # training loop
    print()
    lags, lags_per_client, all_grad_norms = [], [], []
    for current_round in tqdm.tqdm(range(args.num_round)):
        # sample clients which are involved in this round
        sample_clients = np.random.choice(clients, num_sample_client, replace = False)
        
        lag, grad_norms, grad_norms_per_para = 0, [], []
        for c in sample_clients:
            # client side
            c.model.load_state_dict(latest_client_model.state_dict())
            x, y = c.forward_one_batch()
            c.to('cpu')
            
            # server side
            server.to(args.device)
            server.freeze(False)
            server.model.train()
            start = time.perf_counter()
            for current_server_epoch in range(args.num_server_epoch):
                server.optim.zero_grad()
                preds = server.model(x)
                if args.binary is None: # regression, for openeds2020
                    loss = (1 - F.cosine_similarity(preds, y)).mean()
                else: # classification, no matter True of False
                    loss = F.cross_entropy(preds, y)
                loss.backward()
                server.optim.step()
            
            server.model.zero_grad()
            server.freeze(True)
            
            end = time.perf_counter()
            lag += end - start
            
            grad, elapsed = server.grad_one_batch(x, y)
            lag += elapsed
            
            # log grad norm
            g_norm, g_norm_per_para = grad_norm(grad)
            grad_norms.append(g_norm)
            grad_norms_per_para.append(g_norm_per_para)
            all_grad_norms.append(g_norm.cpu())
            
            # client side
            c.to(args.device)
            c.backward_one_batch(grad)
            c.apply_optim()
            latest_client_model.load_state_dict(c.model.state_dict())
            
        # log lag and grad norm
        wandb_log = {}
        lags.append(lag)
        lags_per_client.append(lag / num_sample_client)
        wandb_log['revision/latency'] = lag
        wandb_log['revision/latency_per_client'] = lag / num_sample_client
        wandb_log['revision/grad_norm'] = sum(grad_norms) / len(grad_norms)
        wandb_log['revision/grad_norm_per_para'] = sum(grad_norms_per_para) / len(grad_norms_per_para)
        
        # evaluation
        eval_help(args, sample_clients, server, wandb_log)
        
    # wandb log at the end of training
    wandb_log = {}
    wandb_log['revision/sum_latency'] = sum(lags)
    wandb_log['revision/avg_latency'] = sum(lags) / len(lags)
    wandb_log['revision/std_latency'] = np.std(lags)
    wandb_log['revision/avg_latency_per_client'] = sum(lags_per_client) / len(lags_per_client)
    wandb_log['revision/std_latency_per_client'] = np.std(lags_per_client)
    wandb_log['revision/avg_grad_norm'] = sum(all_grad_norms) / len(all_grad_norms)
    wandb_log['revision/std_grad_norm'] = np.std(all_grad_norms)
    wandb.log(wandb_log)

######################################## alias functions ########################################
def SL(args: object, clients: list[object], server: object, num_sample_client: int) -> None:
    SSL(args, clients, server, num_sample_client)

def FSL(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    SFLV1(args, clients, server, num_sample_client, agg_all_clients)

def SplitFedV1(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    SFLV1(args, clients, server, num_sample_client, agg_all_clients)

def SplitFedV2(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    SFLV2(args, clients, server, num_sample_client, agg_all_clients)

def FL(args: object, clients: list[object], server: object, num_sample_client: int, agg_all_clients: bool = True) -> None:
    FedAvg(args, clients, server, num_sample_client, agg_all_clients)

def CycleFSL(args: object, clients: list[object], server: object, num_sample_client: int) -> None:
    CycleSFL(args, clients, server, num_sample_client)