/*
 * Globally-Optimal Gaussian Mixture Alignment (GOGMA): config_map
 * Open and parse a config file.
 * 
 * Campbell, D., and Petersson, L., "GOGMA: Globally-Optimal Gaussian
 * Mixture Alignment", IEEE Conference on Computer Visision and
 * Pattern Recognition (CVPR), Las Vegas, USA, IEEE, Jun. 2016
 * 
 * For the full license, see license.txt in the root directory
 * 
 * Author: Dylan Campbell
 * Date: 20160212
 * Revision: 1.1
 */

#include "config_map.h"

ConfigMap::ConfigMap() {
};

ConfigMap::ConfigMap(const char * config_file) {
  std::ifstream fin(config_file);
  
  if (!fin.is_open()) {
    std::cout << "Unable to open config file '" << config_file << "'" << std::endl;
    exit(-2);
  } else {
    std::string buffer;
    while (std::getline(fin, buffer)) {
    	// Ignore comments
      if (buffer.c_str()[0] == '#') {
      } else {
        this->addLine(buffer);
      }
    }
    fin.close();
  }
};

ConfigMap::~ConfigMap() {
  for (std::list<double*>::iterator iter = double_allocated_memory_collector.begin();
       iter != double_allocated_memory_collector.end(); iter++) {
    delete [] (*iter);
  }
  for (std::list<int*>::iterator iter = int_allocated_memory_collector.begin();
       iter != int_allocated_memory_collector.end(); iter++) {
    delete [] (*iter);
  }
};

void ConfigMap::addLine(std::string line_) {
  // Swallow last character (carriage return: ASCII 13)
  if (line_.size() > 0) {
    if ((int)line_.c_str()[line_.size() - 1] == 13) {
      line_.resize (line_.size () - 1);
    }
  }

  StringTokenizer st(line_, const_cast<char*>(" =;"));
  if (st.numberOfTokens() != 2) {
    return;
  }

  std::string key = st.nextToken();
  std::string val = st.nextToken();
  addPair(key, val);
};

void ConfigMap::addPair(std::string key_, std::string value_) {
  mappings[key_] = value_;
};

char * ConfigMap::get(char * key_) {
  std::string key(key_);
#ifdef VERBOSE
  std::cout << "DEBUG::ConfigMap.get()::key is `" << key_ << "'" << std::endl;
  std::string val = mappings[key];
  std::cout << "DEBUG::Requesting (" << key_ << ")->(" << val << ")" << std::endl;
  return const_cast<char*>(val.c_str());
#else
  return const_cast<char*>(mappings[key].c_str());
#endif
};

char * ConfigMap::get(const char * key_) {
  return get(const_cast<char*>(key_));
};

int ConfigMap::getI(char * key_) {
  char * str_val = get(key_);
  
  if (str_val == NULL) {
    return 0;
  } else {
    return atoi(str_val);
  } 
};

int ConfigMap::getI(const char * key_) {
  return getI(const_cast<char*>(key_));
};

double ConfigMap::getF(char *key_) {
  char * str_val = get(key_);
  
  if (str_val == NULL) {
    return 0;
  } else {
    return atof(str_val);
  }
};

double ConfigMap::getF(const char * key_) {
  return getF(const_cast<char*>(key_));
};

int* ConfigMap::getIArray(char * key_) {
  std::string key(key_);
  std::string val = mappings[key];
  if (val.empty()) {
    return NULL;
  }
  StringTokenizer st(val, const_cast<char*>("(,)"));
  int* return_array = new int[st.numberOfTokens()];
  int_allocated_memory_collector.push_back(return_array);
  for (int i = 0; st.numberOfTokens() > 0; i++) {
    return_array[i] = (int)atoi(st.nextToken().c_str());
  }
  return return_array;
};

int* ConfigMap::getIArray(const char * key_) {
  return getIArray(const_cast<char*>(key_));
};

int* ConfigMap::getIArray(char * key_, int& num_elements_) {
  std::string key(key_);
  std::string val = mappings[key];
  if (val.empty()) {
    return NULL;
  }
  StringTokenizer st(val, const_cast<char*>("(,)"));
  int* return_array = new int[st.numberOfTokens()];
  int_allocated_memory_collector.push_back(return_array);
  for (int i = 0; st.numberOfTokens() > 0; i++) {
    return_array[i] = (int)atoi(st.nextToken().c_str());
  }
  num_elements_ = st.numberOfTokens();
  return return_array;
};

int* ConfigMap::getIArray(const char * key_, int& num_elements_) {
  return getIArray(const_cast<char*>(key_), num_elements_);
};

double* ConfigMap::getFArray(char * key_) {
  std::string key(key_);
  std::string val = mappings[key];
  if (val.empty()) {
    return NULL;
  }
  StringTokenizer st(val, const_cast<char*>("(,)"));
  double* return_array = new double[st.numberOfTokens()];
  double_allocated_memory_collector.push_back(return_array);
  for (int i = 0; st.numberOfTokens() > 0; i++) {
    return_array[i] = (double)atof(st.nextToken().c_str());
  }
  return return_array;
};

double* ConfigMap::getFArray(const char * key_) {
  return getFArray(const_cast<char*>(key_));
};

double* ConfigMap::getFArray(char * key_, int& num_elements_) {
  std::string key(key_);
  std::string val = mappings[key];
  if (val.empty()) {
    return NULL;
  }
  StringTokenizer st(val, const_cast<char*>("(,)"));
  double* return_array = new double[st.numberOfTokens()];
  double_allocated_memory_collector.push_back(return_array);
  for (int i = 0; st.numberOfTokens() > 0; i++) {
    return_array[i] = (double)atof(st.nextToken().c_str());
  }
  num_elements_ = st.numberOfTokens();
  return return_array;
};

double* ConfigMap::getFArray(const char * key_, int& num_elements_) {
  return getFArray(const_cast<char*>(key_), num_elements_);
};

void ConfigMap::print() {
  for (std::map<std::string,std::string>::const_iterator iter = mappings.begin(); iter != mappings.end(); iter++)
  {
    std::cout << "(" << iter->first << ")->(" << iter->second << ")" << std::endl;
  }
};
